# -*- coding: utf-8 -*-

from __future__ import division

__copyright__ = "Copyright (C) 2014 Andreas Kloeckner"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""


import six
import django.forms as forms
from djangosaml2.backends import Saml2Backend as Saml2BackendBase


class StyledForm(forms.Form):
    def __init__(self, *args, **kwargs):
        from crispy_forms.helper import FormHelper
        self.helper = FormHelper()
        self.helper.form_class = "form-horizontal"
        self.helper.label_class = "col-lg-2"
        self.helper.field_class = "col-lg-8"

        super(StyledForm, self).__init__(*args, **kwargs)


class StyledInlineForm(forms.Form):
    def __init__(self, *args, **kwargs):
        from crispy_forms.helper import FormHelper
        self.helper = FormHelper()
        self.helper.form_class = "form-inline"
        self.helper.label_class = "sr-only"

        super(StyledInlineForm, self).__init__(*args, **kwargs)


class StyledModelForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        from crispy_forms.helper import FormHelper
        self.helper = FormHelper()
        self.helper.form_class = "form-horizontal"
        self.helper.label_class = "col-lg-2"
        self.helper.field_class = "col-lg-8"

        super(StyledModelForm, self).__init__(*args, **kwargs)


# {{{ maintenance mode

def is_maintenance_mode(request):
    from django.conf import settings
    maintenance_mode = getattr(settings, "RELATE_MAINTENANCE_MODE", False)

    if maintenance_mode:
        exceptions = getattr(settings, "RELATE_MAINTENANCE_MODE_EXCEPTIONS", [])

        import ipaddress

        remote_address = ipaddress.ip_address(
                six.text_type(request.META['REMOTE_ADDR']))

        for exc in exceptions:
            if remote_address in ipaddress.ip_network(six.text_type(exc)):
                maintenance_mode = False
                break

    return maintenance_mode


class MaintenanceMiddleware(object):
    def process_request(self, request):
        if is_maintenance_mode(request):
            from django.shortcuts import render
            return render(request, "maintenance.html")

# }}}


def settings_context_processor(request):
    from django.conf import settings
    return {
        "student_sign_in_view": "relate-sign_in_choice",
        "relate_sign_in_by_email_enabled":
        settings.RELATE_SIGN_IN_BY_EMAIL_ENABLED,
        "relate_registration_enabled":
        settings.RELATE_REGISTRATION_ENABLED,
        "relate_sign_in_by_exam_tickets_enabled":
        settings.RELATE_SIGN_IN_BY_EXAM_TICKETS_ENABLED,
        "relate_sign_in_by_saml2_enabled":
        settings.RELATE_SIGN_IN_BY_SAML2_ENABLED,
        "maintenance_mode": is_maintenance_mode(request),
        "site_announcement": getattr(settings, "RELATE_SITE_ANNOUNCEMENT", None),
        }


def as_local_time(datetime):
    """Takes an timezone-aware datetime and applies the server timezone."""
    from django.conf import settings
    from pytz import timezone
    tz = timezone(settings.TIME_ZONE)
    return datetime.astimezone(tz)


def localize_datetime(datetime):
    """Takes an timezone-naive datetime and applies the server timezone."""
    from django.conf import settings
    from pytz import timezone
    tz = timezone(settings.TIME_ZONE)
    return tz.localize(datetime)


def local_now():
    from django.conf import settings
    from pytz import timezone
    tz = timezone(settings.TIME_ZONE)
    from datetime import datetime
    return tz.localize(datetime.now())


def format_datetime_local(datetime, format='DATETIME_FORMAT'):
    """
    Format a datetime object to a localized string via python.

    Note: The datetime rendered in template is itself locale aware.
    A custom format must be defined in settings.py.
    When a custom format uses a same name with an existing built-in
    format, it will be overrided by built-in format if l10n
    is enabled.
    """

    fmt = format

    from django.utils import formats
    from django.utils.dateformat import format

    try:
        return formats.date_format(datetime, fmt)
    except AttributeError:
        try:
            return format(datetime, fmt)
        except AttributeError:
            return formats.date_format(datetime, "DATETIME_FORMAT")


# {{{ dict_to_struct

class Struct(object):
    def __init__(self, entries):
        for name, val in six.iteritems(entries):
            self.__dict__[name] = val

    def __repr__(self):
        return repr(self.__dict__)


def dict_to_struct(data):
    if isinstance(data, list):
        return [dict_to_struct(d) for d in data]
    elif isinstance(data, dict):
        return Struct({k: dict_to_struct(v) for k, v in six.iteritems(data)})
    else:
        return data


def struct_to_dict(data):
    return dict(
            (name, val)
            for name, val in six.iteritems(data.__dict__)
            if not name.startswith("_"))

# }}}


def retry_transaction(f, args, kwargs={}, max_tries=None, serializable=None):
    from django.db import transaction
    from django.db.utils import OperationalError

    if max_tries is None:
        max_tries = 5
    if serializable is None:
        serializable = False

    assert max_tries > 0
    while True:
        try:
            with transaction.atomic():
                if serializable:
                    from django.db import connections, DEFAULT_DB_ALIAS
                    conn = connections[DEFAULT_DB_ALIAS]
                    if conn.vendor == "postgresql":
                        cursor = conn.cursor()
                        cursor.execute(
                                "SET TRANSACTION ISOLATION LEVEL SERIALIZABLE;")

                return f(*args, **kwargs)
        except OperationalError:
            max_tries -= 1
            if not max_tries:
                raise


class retry_transaction_decorator(object):
    def __init__(self, max_tries=None, serializable=None):
        self.max_tries = max_tries
        self.serializable = serializable

    def __call__(self, f):
        from functools import update_wrapper

        def wrapper(*args, **kwargs):
            return retry_transaction(f, args, kwargs,
                    max_tries=self.max_tries,
                    serializable=self.serializable)

        update_wrapper(wrapper, f)
        return wrapper


# {{{ hang debugging

def dumpstacks(signal, frame):
    import threading
    import sys
    import traceback

    id2name = dict([(th.ident, th.name) for th in threading.enumerate()])
    code = []
    for threadId, stack in sys._current_frames().items():
        code.append("\n# Thread: %s(%d)" % (id2name.get(threadId, ""), threadId))
        for filename, lineno, name, line in traceback.extract_stack(stack):
            code.append('File: "%s", line %d, in %s' % (filename, lineno, name))
            if line:
                code.append("  %s" % (line.strip()))
    print("\n".join(code))

if 0:
    import signal
    import os
    print("*** HANG DUMP HANDLER ACTIVATED: 'kill -USR1 %s' to dump stacks"
            % os.getpid())
    signal.signal(signal.SIGUSR1, dumpstacks)

# }}}


# {{{ convert django language name to js styled language name

def to_js_lang_name(dj_lang_name):
    """
    Turns a django language name (en-us) into a js styled language
    name (en-US).
    """
    p = dj_lang_name.find('-')
    if p >= 0:
        return dj_lang_name[:p].lower() + '-' + dj_lang_name[p + 1:].upper()
    else:
        return dj_lang_name.lower()

# }}}


# {{{ SAML auth backend

# This ticks the 'verified' boxes once we've receive attribute assertions
# through SAML2.

class Saml2Backend(Saml2BackendBase):
    def _set_attribute(self, obj, attr, value):
        mod = super(Saml2Backend, self)._set_attribute(obj, attr, value)

        if attr == "institutional_id":
            if not obj.institutional_id_verified:
                obj.institutional_id_verified = True
                mod = True

        if attr in ["first_name", "last_name"]:
            if not obj.name_verified:
                obj.name_verified = True
                mod = True

        return mod

# }}}

# vim: foldmethod=marker
