__copyright__ = "Copyright (C) 2010-2013 Andreas Kloeckner"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

from collections.abc import Callable, Iterable
from dataclasses import field
from warnings import warn
from functools import partial
from typing import Any, Concatenate, Literal, TypeAlias, TypeVar

import numpy as np

import modepy as mp
from pymbolic import ExpressionNode, Variable
from pymbolic.primitives import (  # noqa: N813
        Variable as var,
        cse_scope as cse_scope_base,
        make_common_subexpression as cse,
        expr_dataclass)
from pymbolic.geometric_algebra import MultiVector, componentwise
from pymbolic.geometric_algebra.primitives import (
        NablaComponent, Derivative as DerivativeBase)
from pymbolic.primitives import make_sym_vector
from pymbolic.typing import ArithmeticExpression

from pytools import P
from pytools.obj_array import make_obj_array, flat_obj_array
from sumpy.kernel import Kernel, SpatialConstant

from pytential.symbolic.dof_desc import (
        DEFAULT_SOURCE, DEFAULT_TARGET, DEFAULT_DOFDESC,
        QBX_SOURCE_STAGE1, QBX_SOURCE_STAGE2, QBX_SOURCE_QUAD_STAGE2,
        GRANULARITY_NODE, GRANULARITY_CENTER, GRANULARITY_ELEMENT,
        DOFDescriptor, DOFDescriptorLike,
        as_dofdesc,
        )

__doc__ = """
.. |dofdesc-blurb| replace:: A
    :class:`~pytential.symbolic.dof_desc.DOFDescriptor` or a symbolic
    name for a geometric object (such as a
    :class:`~meshmode.discretization.Discretization`).


Object types
^^^^^^^^^^^^
Based on the mathematical quantity being represented, the following types of
objects occur as part of a symbolic operator representation:

*   If a quantity is a scalar, it is just a symbolic expression--that is, an
    element of the set of formal expressions recursively generated by the
    placeholders (see :ref:`placeholders`), constants, and arithmetic operations
    on them (see :mod:`pymbolic.primitives`).  Objects of this type are created
    simply by doing arithmetic on placeholders and scalar constants.

*   If the quantity is "just a bunch of scalars" (like, say, rows in a system
    of integral equations), the symbolic representation is an object array. Each
    element of the object array contains a symbolic expression.

    :func:`pytools.obj_array.make_obj_array` and
    :func:`pytools.obj_array.flat_obj_array`
    can help create those.

*   If it is a geometric quantity (that makes sense without explicit reference to
    coordinates), it is a :class:`pymbolic.geometric_algebra.MultiVector`.
    This can be converted to an object array by calling:
    :meth:`pymbolic.geometric_algebra.MultiVector.as_vector`.

:class:`pyopencl.array.Array` and :class:`meshmode.dof_array.DOFArray` instances
hold per-node degrees of freedom (and only those). Such instances do *not* occur
on the symbolic side of :mod:`pytential` at all. They're only visible either as
bound inputs (see :func:`pytential.bind`) or outputs of evaluation. Which one is
used depends on the meaning of the data being represented. If the data is
associated with a :class:`~meshmode.discretization.Discretization`, then
:class:`~meshmode.dof_array.DOFArray` is used and otherwise
:class:`~pyopencl.array.Array` is used.

.. autoclass:: Expression
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: for_each_expression

.. autoclass:: Operand

.. autoclass:: ArithmeticExpressionT

.. autoclass:: QBXForcedLimit

.. class:: P

    See :class:`pytools.P`

.. class:: ExpressionNode

    See :class:`pymbolic.ExpressionNode`.

Diagnostics
^^^^^^^^^^^

.. autoclass:: ErrorExpression
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. _placeholders:

Placeholders
^^^^^^^^^^^^

.. autoclass:: var

.. autoclass:: SpatialConstant
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: make_sym_mv
.. autofunction:: make_sym_surface_mv

Functions
^^^^^^^^^

.. data:: real
.. data:: imag
.. data:: conj
.. data:: abs

.. data:: sqrt

.. data:: sin
.. data:: cos
.. data:: tan

.. data:: arcsin
.. data:: arccos
.. data:: arctan
.. data:: arctan2

.. data:: sinh
.. data:: cosh
.. data:: tanh

.. data:: arcsinh
.. data:: arccosh
.. data:: arctanh

.. data:: exp
.. data:: log

Discretization properties
^^^^^^^^^^^^^^^^^^^^^^^^^

.. autoclass:: DiscretizationProperty
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autoclass:: IsShapeClass
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autoclass:: QWeight
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: nodes
.. autofunction:: parametrization_derivative
.. autofunction:: parametrization_derivative_matrix
.. autofunction:: pseudoscalar
.. autofunction:: area_element
.. autofunction:: sqrt_jac_q_weight
.. autofunction:: normal
.. autofunction:: mean_curvature
.. autofunction:: first_fundamental_form
.. autofunction:: second_fundamental_form
.. autofunction:: shape_operator

.. autofunction:: expansion_radii
.. autofunction:: expansion_centers
.. autofunction:: h_max
.. autofunction:: weights_and_area_elements

Elementary numerics
^^^^^^^^^^^^^^^^^^^

.. autoclass:: NumReferenceDerivative
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: num_reference_derivative

.. autoclass:: NodeSum
    :undoc-members:
    :members: mapper_method

.. autofunction:: node_sum

.. autoclass:: NodeMax
    :undoc-members:
    :members: mapper_method

.. autofunction:: node_max

.. autoclass:: NodeMin
    :undoc-members:
    :members: mapper_method

.. autofunction:: node_min

.. autoclass:: ElementwiseSum
    :undoc-members:
    :members: mapper_method

.. autofunction:: elementwise_sum

.. autoclass:: ElementwiseMin
    :undoc-members:
    :members: mapper_method

.. autofunction:: elementwise_min

.. autoclass:: ElementwiseMax
    :undoc-members:
    :members: mapper_method

.. autofunction:: elementwise_max

.. autofunction:: integral

.. autoclass:: Ones
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: ones_vec
.. autofunction:: area
.. autofunction:: mean

.. autoclass:: IterativeInverse
    :show-inheritance:
    :undoc-members:
    :members: mapper_method


Operators
^^^^^^^^^

.. autoclass:: Interpolation
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: interpolate

Geometric Calculus (based on Geometric/Clifford Algebra)
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

.. autoclass:: Derivative
    :undoc-members:

Conventional Calculus
^^^^^^^^^^^^^^^^^^^^^

.. autofunction:: dd_axis
.. function:: d_dx
.. function:: d_dy
.. function:: d_dz
.. autofunction:: grad_mv
.. autofunction:: grad
.. autofunction:: laplace

Layer potentials
^^^^^^^^^^^^^^^^

.. autoclass:: IntG
    :show-inheritance:
    :undoc-members:
    :members: mapper_method

.. autofunction:: int_g_dsource
.. autofunction:: int_g_vec

.. autofunction:: S
.. autofunction:: Sp
.. autofunction:: Spp
.. autofunction:: D
.. autofunction:: Dp
.. autofunction:: normal_derivative
.. autofunction:: tangential_derivative

"Conventional" Vector Calculus
^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^

.. autofunction:: tangential_onb
.. autofunction:: xyz_to_tangential
.. autofunction:: tangential_to_xyz
.. autofunction:: project_to_tangential
.. autofunction:: cross
.. autofunction:: n_dot
.. autofunction:: n_cross
.. autofunction:: curl

Pretty-printing expressions
^^^^^^^^^^^^^^^^^^^^^^^^^^^

.. autofunction:: pretty
"""

__all__ = (
    "Expression",
    "Operand",
    "ArithmeticExpressionT",
    "QBXForcedLimit",
    "for_each_expression",

    "ErrorExpression",

    "var", "SpatialConstant", "make_sym_mv", "make_sym_surface_mv",

    "real", "imag", "conj", "abs",
    "sqrt",
    "sin", "cos", "tan", "arcsin", "arccos", "arctan", "arctan2",
    "sinh", "cosh", "tanh", "arcsinh", "arccosh", "arctanh",
    "exp", "log",

    "IsShapeClass", "QWeight", "nodes", "parametrization_derivative",
    "parametrization_derivative_matrix", "pseudoscalar", "area_element",
    "sqrt_jac_q_weight", "normal", "mean_curvature",
    "first_fundamental_form", "second_fundamental_form", "shape_operator",

    "expansion_radii", "expansion_centers", "h_max", "weights_and_area_elements",

    "NumReferenceDerivative", "NodeSum", "NodeMax", "NodeMin", "ElementwiseSum",
    "ElementwiseMax", "integral", "Ones", "ones_vec", "area", "mean",
    "IterativeInverse",

    "Interpolation", "interpolate",

    "Derivative",

    "dd_axis", "d_dx", "d_dy", "d_dz", "grad_mv", "grad", "laplace",


    "IntG", "int_g_dsource", "int_g_vec",

    "S", "Sp", "Spp", "D", "Dp",

    "normal_derivative", "tangential_derivative",

    "tangential_onb", "xyz_to_tangential", "tangential_to_xyz",
    "project_to_tangential",
    "cross", "n_dot", "n_cross", "curl",

    "pretty",

    # re-exported symbols, maybe questionable
    "flat_obj_array",
    "DEFAULT_SOURCE", "DEFAULT_TARGET",
    "QBX_SOURCE_STAGE1", "QBX_SOURCE_STAGE2", "QBX_SOURCE_QUAD_STAGE2",
    "GRANULARITY_NODE", "GRANULARITY_CENTER", "GRANULARITY_ELEMENT",
    "DOFDescriptor", "DOFDescriptorLike", "as_dofdesc",
    )


# {{{ helpers


@expr_dataclass()
class Expression(ExpressionNode):
    """A subclass of :class:`pymbolic.primitives.ExpressionNode` for use with
    :mod:`pytential` mappers.
    """

    def make_stringifier(self, originating_stringifier=None):
        from pytential.symbolic.mappers import StringifyMapper
        return StringifyMapper()


Operand: TypeAlias = (
    ArithmeticExpression | np.ndarray[Any, np.dtype[Any]] | MultiVector)
QBXForcedLimit: TypeAlias = int | Literal["avg"] | None

# NOTE: this will likely live in pymbolic at some point, but for now we take it!
ArithmeticExpressionT = TypeVar("ArithmeticExpressionT", bound=ArithmeticExpression)


class _NoArgSentinel:
    pass


class cse_scope(cse_scope_base):  # noqa: N801
    DISCRETIZATION = "pytential_discretization"


def for_each_expression(
        f: Callable[Concatenate[ArithmeticExpression, P], ArithmeticExpression]
        ) -> Callable[Concatenate[Operand, P], Operand]:
    """A decorator that takes a function that can only work on expressions
    and transforms it into a function that can be applied componentwise on
    :class:`numpy.ndarray` or :class:`~pymbolic.geometric_algebra.MultiVector`.
    """

    from functools import wraps

    @wraps(f)
    def wrapper(operand: Operand, *args: P.args, **kwargs: P.kwargs) -> Operand:
        if isinstance(operand, np.ndarray | MultiVector):
            def func(operand_i: ArithmeticExpression) -> ArithmeticExpression:
                return f(operand_i, *args, **kwargs)

            return componentwise(func, operand)
        else:
            return f(operand, *args, **kwargs)

    return wrapper

# }}}


@expr_dataclass()
class NamedIntermediateResult(Variable):
    """Internal variables used by ``pytential.compiler``."""


@expr_dataclass()
class ErrorExpression(Expression):
    """An expression that, if evaluated, causes an error with the supplied
    *message*.

    .. autoattribute:: message
    """

    message: str
    """The error message to raise when this expression is encountered."""


def make_sym_mv(name: str, num_components: int) -> MultiVector[Expression]:
    return MultiVector(make_sym_vector(name, num_components))


def make_sym_surface_mv(name, ambient_dim, dim, dofdesc=None):
    par_grad = parametrization_derivative_matrix(ambient_dim, dim, dofdesc)

    return sum(
            var(f"{name}{i}")
            * cse(MultiVector(vec), f"tangent{i}", cse_scope.DISCRETIZATION)
            for i, vec in enumerate(par_grad.T))


@expr_dataclass()
class Function(Variable):
    def __call__(self, operand, *args, **kwargs):
        # If the call is handed an object array full of operands,
        # return an object array of the operator applied to each of the
        # operands.

        from pytools.obj_array import obj_array_vectorize
        if isinstance(operand, np.ndarray) and operand.dtype.char == "O":
            def make_op(operand_i):
                return self(operand_i, *args, **kwargs)

            return obj_array_vectorize(make_op, operand)
        else:
            return var.__call__(self, operand, *args, **kwargs)


@expr_dataclass()
class NumpyMathFunction(Function):
    """A math function named within the numpy naming convention and with
    numpy-like semantics."""


real = NumpyMathFunction("real")
imag = NumpyMathFunction("imag")
conj = NumpyMathFunction("conj")
abs = NumpyMathFunction("abs")

sqrt = NumpyMathFunction("sqrt")

sin = NumpyMathFunction("sin")
cos = NumpyMathFunction("cos")
tan = NumpyMathFunction("tan")

arcsin = NumpyMathFunction("arcsin")
arccos = NumpyMathFunction("arccos")
arctan = NumpyMathFunction("arctan")
arctan2 = NumpyMathFunction("arctan2")

sinh = NumpyMathFunction("sinh")
cosh = NumpyMathFunction("cosh")
tanh = NumpyMathFunction("tanh")

arcsinh = NumpyMathFunction("arcsinh")
arccosh = NumpyMathFunction("arccosh")
arctanh = NumpyMathFunction("arctanh")

exp = NumpyMathFunction("exp")
log = NumpyMathFunction("log")


# {{{ discretization properties

@expr_dataclass()
class DiscretizationProperty(Expression):
    """A quantity that depends exclusively on the discretization.

    .. autoattribute:: dofdesc
    """

    dofdesc: DOFDescriptor
    """The descriptor for this quantity that selects its geometry on evaluation."""

    def __post_init__(self) -> None:
        if not isinstance(self.dofdesc, DOFDescriptor):
            warn("Passing a 'dofdesc' that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "dofdesc", as_dofdesc(self.dofdesc))


@expr_dataclass(init=False)
class IsShapeClass(DiscretizationProperty):
    """A predicate that is *True* if the elements of the discretization have
    a unique type that matches :attr:`shape`.

    .. autoattribute:: shape
    """

    shape: mp.Shape
    """A :class:`modepy.Shape` subclass."""

    # FIXME: this is added for backwards compatibility with pre-dataclass expressions
    def __init__(self, shape: mp.Shape, dofdesc: DOFDescriptorLike) -> None:
        object.__setattr__(self, "shape", shape)
        super().__init__(dofdesc)  # type: ignore[arg-type]


@expr_dataclass()
class QWeight(DiscretizationProperty):
    """Bare quadrature weights (without Jacobians)."""


@expr_dataclass(init=False)
class NodeCoordinateComponent(DiscretizationProperty):
    """
    .. autoattribute:: ambient_axis
    """

    ambient_axis: int
    """The axis index this node coordinate represents, i.e. 0 for $x$, etc."""

    # FIXME: this is added for backwards compatibility with pre-dataclass expressions
    def __init__(self,
                 ambient_axis: int,
                 dofdesc: DOFDescriptorLike | None = None) -> None:
        object.__setattr__(self, "ambient_axis", ambient_axis)
        super().__init__(dofdesc)   # type: ignore[arg-type]


def nodes(ambient_dim, dofdesc=None):
    """Return a :class:`pymbolic.geometric_algebra.MultiVector` of node
    locations.
    """
    dofdesc = as_dofdesc(dofdesc)
    return MultiVector(
            make_obj_array([
                NodeCoordinateComponent(i, dofdesc)
                for i in range(ambient_dim)]))


@expr_dataclass()
class NumReferenceDerivative(DiscretizationProperty):
    """An operator that takes a derivative of *operand* with respect to the the
    element reference coordinates.

    .. autoattribute:: ref_axes
    .. autoattribute:: operand
    """

    ref_axes: tuple[tuple[int, int], ...]
    """A tuple of pairs ``(axis, derivative_order)`` that define the reference
    derivatives taken on the given *operand*. The tuple must be sorted with
    respect to the axis index. For example, ``((0, 2), (1, 1))`` is a correct
    input as it is sorted and each axis is unique. It denotes a second derivative
    with respect to $x$ (0) and a first derivative with respect to $y$ (1).
    """
    operand: Operand
    """An operand to differentiate."""

    def __new__(cls,
                ref_axes: int | tuple[tuple[int, int], ...] | None = None,
                operand: Operand | None = None,
                dofdesc: DOFDescriptor | None = None,
                ) -> "NumReferenceDerivative":
        if isinstance(operand, np.ndarray | MultiVector):
            warn(f"Passing {type(operand)} directly to {cls.__name__!r} "
                 "is deprecated and will result in an error from 2025. Use "
                 "the 'num_reference_derivative' function instead.",
                 DeprecationWarning, stacklevel=3)

            def make_op(operand_i):
                return cls(ref_axes, operand_i, as_dofdesc(dofdesc))

            # FIXME: mypy is right: new should return `cls` instances and we're
            # abusing it to vectorize the call like this.
            return componentwise(make_op, operand)  # type: ignore[return-value]
        else:
            return DiscretizationProperty.__new__(cls)

    # FIXME: this is added for backwards compatibility with pre-dataclass expressions.
    # Ideally, we'd just have a __post_init__, but the order of the arguments is
    # different..
    def __init__(self,
                 ref_axes: tuple[tuple[int, int], ...],
                 operand: ArithmeticExpression,
                 dofdesc: DOFDescriptorLike) -> None:
        if isinstance(ref_axes, int):
            warn(f"Passing an 'int' as 'ref_axes' to {type(self).__name__!r} "
                 "is deprecated and will result in an error in 2025. Pass the "
                 "well-formatted tuple '((ref_axes, 1),)' instead.",
                 DeprecationWarning, stacklevel=2)

            ref_axes = ((ref_axes, 1),)

        if not isinstance(ref_axes, tuple):
            raise ValueError(f"'ref_axes' must be a tuple: {type(ref_axes)}")

        if tuple(sorted(ref_axes)) != ref_axes:
            raise ValueError(
                f"'ref_axes' must be sorted by axis index: {ref_axes}"
            )

        if len(dict(ref_axes)) != len(ref_axes):
            raise ValueError(
                f"'ref_axes' must not contain an axis more than once: {ref_axes}"
            )

        object.__setattr__(self, "ref_axes", ref_axes)
        object.__setattr__(self, "operand", operand)
        super().__init__(dofdesc)   # type: ignore[arg-type]


@for_each_expression
def num_reference_derivative(
        expr: ArithmeticExpression,
        ref_axes: tuple[tuple[int, int], ...] | int,
        dofdesc: DOFDescriptorLike | None) -> NumReferenceDerivative:
    """Take a derivative of *expr* with respect to the the element reference
    coordinates.

    See :class:`~pytential.symbolic.primitives.NumReferenceDerivative`.
    """

    if isinstance(ref_axes, int):
        ref_axes = ((ref_axes, 1),)

    return NumReferenceDerivative(ref_axes, expr, as_dofdesc(dofdesc))


def reference_jacobian(func, output_dim, dim, dofdesc=None):
    """Return a :class:`numpy.ndarray` representing the Jacobian of a vector function
    with respect to the reference coordinates.
    """
    dofdesc = as_dofdesc(dofdesc)
    jac = np.zeros((output_dim, dim), object)

    for i in range(output_dim):
        func_component = func[i]
        for j in range(dim):
            jac[i, j] = num_reference_derivative(func_component, j, dofdesc)

    return jac


def parametrization_derivative_matrix(ambient_dim, dim, dofdesc=None):
    """Return a :class:`numpy.ndarray` representing the derivative of the
    reference-to-global parametrization.
    """

    dofdesc = as_dofdesc(dofdesc)
    return cse(
            reference_jacobian(
                [NodeCoordinateComponent(i, dofdesc) for i in range(ambient_dim)],
                ambient_dim, dim, dofdesc=dofdesc),
            "pd_matrix", cse_scope.DISCRETIZATION)


def parametrization_derivative(ambient_dim, dim, dofdesc=None):
    """Return a :class:`pymbolic.geometric_algebra.MultiVector` representing
    the derivative of the reference-to-global parametrization.
    """

    par_grad = parametrization_derivative_matrix(ambient_dim, dim, dofdesc)

    from pytools import product
    return product(MultiVector(vec) for vec in par_grad.T)


def pseudoscalar(ambient_dim, dim=None, dofdesc=None):
    """
    Same as the outer product of all parametrization derivative columns.
    """
    if dim is None:
        dim = ambient_dim - 1

    return cse(
            parametrization_derivative(ambient_dim, dim, dofdesc)
            .project_max_grade(),
            "pseudoscalar", cse_scope.DISCRETIZATION)


def area_element(ambient_dim, dim=None, dofdesc=None):
    return cse(
            sqrt(pseudoscalar(ambient_dim, dim, dofdesc).norm_squared()),
            "area_element", cse_scope.DISCRETIZATION)


def sqrt_jac_q_weight(ambient_dim, dim=None, dofdesc=None):
    dofdesc = as_dofdesc(dofdesc)
    return cse(
            sqrt(
                area_element(ambient_dim, dim, dofdesc)
                * QWeight(dofdesc)),
            "sqrt_jac_q_weight", cse_scope.DISCRETIZATION)


def normal(ambient_dim, dim=None, dofdesc=None):
    """Exterior unit normals."""

    # Don't be tempted to add a sign here. As it is, it produces
    # exterior normals for positively oriented curves and surfaces.

    dofdesc = as_dofdesc(dofdesc)
    pder = (
            pseudoscalar(ambient_dim, dim, dofdesc)
            / area_element(ambient_dim, dim, dofdesc))
    return cse(
            # Dorst Section 3.7.2
            pder << pder.I.inv(),
            "normal",
            scope=cse_scope.DISCRETIZATION)


def mean_curvature(ambient_dim, dim=None, dofdesc=None):
    """(Numerical) mean curvature."""

    if dim is None:
        dim = ambient_dim - 1

    if ambient_dim == 2 and dim == 1:
        # https://en.wikipedia.org/wiki/Curvature#Local_expressions
        xp, yp = parametrization_derivative_matrix(ambient_dim, dim, dofdesc)

        xpp, ypp = cse(
                reference_jacobian([xp[0], yp[0]], ambient_dim, dim, dofdesc),
                "p2d_matrix", cse_scope.DISCRETIZATION)

        kappa = (xp[0]*ypp[0] - yp[0]*xpp[0]) / (xp[0]**2 + yp[0]**2)**(3/2)
    elif ambient_dim == 3 and dim == 2:
        # https://en.wikipedia.org/wiki/Mean_curvature#Surfaces_in_3D_space
        s_op = shape_operator(ambient_dim, dim=dim, dofdesc=dofdesc)
        kappa = -0.5 * sum(s_op[i, i] for i in range(s_op.shape[0]))
    else:
        raise NotImplementedError("not available in {}D for {}D surfaces"
                .format(ambient_dim, dim))

    return kappa


def first_fundamental_form(ambient_dim, dim=None, dofdesc=None):
    if dim is None:
        dim = ambient_dim - 1

    pd_mat = parametrization_derivative_matrix(ambient_dim, dim, dofdesc)
    return cse(np.dot(pd_mat.T, pd_mat), "fundform1")


def second_fundamental_form(ambient_dim, dim=None, dofdesc=None):
    """Compute the second fundamental form of a surface. This is in reference
    to the reference-to-global mapping in use for each element.

    .. note::

        Some references assume that the second fundamental form is computed
        with respect to an orthonormal basis, which this is not.
    """
    if dim is None:
        dim = ambient_dim - 1

    if not (ambient_dim == 3 and dim == 2):
        raise NotImplementedError("only available for surfaces in 3D")

    dofdesc = as_dofdesc(dofdesc)
    r = nodes(ambient_dim, dofdesc=dofdesc).as_vector()

    # https://en.wikipedia.org/w/index.php?title=Second_fundamental_form&oldid=821047433#Classical_notation

    ruu = num_reference_derivative(r, ((0, 2),), dofdesc)
    ruv = num_reference_derivative(r, ((0, 1), (1, 1)), dofdesc)
    rvv = num_reference_derivative(r, ((1, 2),), dofdesc)

    nrml = normal(ambient_dim, dim, dofdesc).as_vector()

    ff2_l = cse(np.dot(ruu, nrml), "fundform2_L")
    ff2_m = cse(np.dot(ruv, nrml), "fundform2_M")
    ff2_n = cse(np.dot(rvv, nrml), "fundform2_N")

    result = np.zeros((2, 2), dtype=object)
    result[0, 0] = ff2_l
    result[0, 1] = result[1, 0] = ff2_m
    result[1, 1] = ff2_n

    return result


def shape_operator(ambient_dim, dim=None, dofdesc=None):
    if dim is None:
        dim = ambient_dim - 1

    if not (ambient_dim == 3 and dim == 2):
        raise NotImplementedError("only available for surfaces in 3D")

    # https://en.wikipedia.org/w/index.php?title=Differential_geometry_of_surfaces&oldid=833587563
    (E, F), (F, G) = first_fundamental_form(ambient_dim, dim, dofdesc)
    (e, f), (f, g) = second_fundamental_form(ambient_dim, dim, dofdesc)

    result = np.zeros((2, 2), dtype=object)
    result[0, 0] = e*G-f*F
    result[0, 1] = f*G-g*F
    result[1, 0] = f*E-e*F
    result[1, 1] = g*E-f*F

    return cse(
            1/(E*G-F*F)*result,
            "shape_operator")


def _element_size(ambient_dim, dim=None, dofdesc=None):
    # A broken quasi-1D approximation of 1D element size. Do not use.
    if dim is None:
        dim = ambient_dim - 1

    dofdesc = as_dofdesc(dofdesc)
    return elementwise_sum(
            area_element(ambient_dim=ambient_dim, dim=dim, dofdesc=dofdesc)
            * QWeight(dofdesc),
            dofdesc)**(1/dim)


def _small_mat_inverse(mat):
    m, n = mat.shape
    if m != n:
        raise ValueError(
                "inverses only make sense for square matrices: "
                f"got a {m}x{n} matrix")

    if m == 1:
        return np.array([[1/mat[0, 0]]], dtype=object)
    elif m == 2:
        (a, b), (c, d) = mat
        return 1/(a*d - b*c) * np.array([
            [d, -b],
            [-c, a],
            ], dtype=object)
    else:
        raise NotImplementedError(f"inverse formula for {m}x{n} matrices")


def _small_mat_eigenvalues(mat):
    m, n = mat.shape
    if m != n:
        raise ValueError(
                "eigenvalues only make sense for square matrices: "
                f"got a {m}x{n} matrix")

    if m == 1:
        return make_obj_array([mat[0, 0]])
    elif m == 2:
        (a, b), (c, d) = mat
        tr_mat = cse(a + d)

        # solutions to lambda**2 - tr(A) * lambda + det(A)
        # NOTE: 4 * b * c + (a - d)**2 can still become negative if the matrix
        # is not positive definite, but there's not much we can do here
        sqrt_discriminant = cse(sqrt(4 * b * c + (a - d)**2))
        return make_obj_array([
            (tr_mat - sqrt_discriminant) / 2,
            (tr_mat + sqrt_discriminant) / 2,
            ])
    else:
        raise NotImplementedError(f"eigenvalue formula for {m}x{n} matrices")


def _small_sym_mat_eigenvalues(mat):
    m, n = mat.shape
    if m != n:
        raise ValueError(
                "eigenvalues only make sense for square matrices: "
                f"got a {m}x{n} matrix")

    if m == 1:
        return make_obj_array([mat[0, 0]])
    elif m == 2:
        (a, b), (_, d) = mat
        tr_mat = cse(a + d)

        # NOTE: solutions to lambda**2 - tr(A) * lambda + det(A)
        # we would normally have
        #       discriminant = tr - 4 * det
        # but that can become negative with some floating point fuzz, so we
        # rewrite it as a sum of squares to avoid that issue
        sqrt_discriminant = cse(sqrt(4 * b**2 + (a - d)**2))
        return make_obj_array([
            (tr_mat - sqrt_discriminant) / 2,
            (tr_mat + sqrt_discriminant) / 2,
            ])
    else:
        raise NotImplementedError(f"eigenvalue formula for {m}x{n} matrices")


def _equilateral_parametrization_derivative_matrix(ambient_dim, dim=None,
        dofdesc=None):
    if dim is None:
        dim = ambient_dim - 1

    pder_mat = parametrization_derivative_matrix(ambient_dim, dim, dofdesc)

    # The above procedure works well only when the 'reference' end of the
    # mapping is in equilateral coordinates.
    from modepy.tools import EQUILATERAL_TO_UNIT_MAP
    equi_to_unit = EQUILATERAL_TO_UNIT_MAP[dim].a

    # This is the Jacobian of the (equilateral reference element) -> (global) map.
    return cse(
            np.dot(pder_mat, equi_to_unit),
            "equilateral_pder_mat")


def _simplex_mapping_max_stretch_factor(ambient_dim, dim=None, dofdesc=None):
    """Return the largest factor by which the reference-to-global
    mapping stretches the bi-unit (i.e. :math:`[-1, 1]`) reference
    element along any axis.

    If *map_elementwise_max* is True, returns a DOF vector that is elementwise
    constant.
    """

    if dim is None:
        dim = ambient_dim - 1

    # The 'technique' here is ad-hoc, but I'm fairly confident it's better than
    # what we had. The idea is that singular values of the mapping Jacobian
    # yield "stretch factors" of the mapping Why? because it maps a right
    # singular vector $`v_1`$ (of unit length) to $`\sigma_1 u_1`$, where
    # $`u_1`$ is the corresponding left singular vector (also of unit length).
    # And so the biggest one tells us about the direction with the 'biggest'
    # stretching, where 'stretching' (*2 to remove bi-unit reference element)
    # reflects available quadrature resolution in that direction.

    equi_pder_mat = _equilateral_parametrization_derivative_matrix(
            ambient_dim, dim, dofdesc)

    # Compute eigenvalues of J^T to compute SVD.
    equi_pder_mat_jtj = cse(
            np.dot(equi_pder_mat.T, equi_pder_mat),
            "pd_mat_jtj")

    stretch_factors = [
            cse(sqrt(s), f"simplex_mapping_singval_{i}")
            # NOTE: multiply by 4 to compensate for equilateral reference
            # elements of side length 2. (J^T J contains two factors of two.)
            for i, s in enumerate(_small_sym_mat_eigenvalues(4 * equi_pder_mat_jtj))
            ]

    from pymbolic.primitives import Max
    return Max(tuple(stretch_factors))


def _hypercube_mapping_max_stretch_factor(ambient_dim, dim=None, dofdesc=None):
    if dim is None:
        dim = ambient_dim - 1

    # NOTE: unlike in the simplex case, here we do not need to transform the
    # reference element, as it already is nicely rotation invariant
    pder_mat = first_fundamental_form(ambient_dim, dim=dim, dofdesc=dofdesc)
    stretch_factors = [
            cse(sqrt(s), f"hypercube_mapping_singval_{i}")
            # NOTE: like for the simplex, we multiply `pder_mat` by 4 to
            # account for the side length of 2 in J^T J
            for i, s in enumerate(_small_sym_mat_eigenvalues(4.0 * pder_mat))
            ]

    from pymbolic.primitives import Max
    return Max(tuple(stretch_factors))


def _mapping_max_stretch_factor(ambient_dim, dim=None, dofdesc=None):
    simplex_stretch_factor = _simplex_mapping_max_stretch_factor(
        ambient_dim, dim, dofdesc=dofdesc)
    hypercube_stretch_factor = _hypercube_mapping_max_stretch_factor(
        ambient_dim, dim, dofdesc=dofdesc)

    import modepy as mp
    from pymbolic.primitives import If
    stretch_factor = If(IsShapeClass(mp.Simplex, dofdesc),
                        simplex_stretch_factor,
                        If(IsShapeClass(mp.Hypercube, dofdesc),
                           hypercube_stretch_factor,
                           ErrorExpression("unknown reference element shape"))
                        )

    return cse(stretch_factor, "mapping_stretch_factor", cse_scope.DISCRETIZATION)


def _max_curvature(ambient_dim, dim=None, dofdesc=None):
    # An attempt at a 'max curvature' criterion.

    if dim is None:
        dim = ambient_dim - 1

    if ambient_dim == 2:
        return abs(mean_curvature(ambient_dim, dim, dofdesc=dofdesc))
    elif ambient_dim == 3:
        shape_op = shape_operator(ambient_dim, dim, dofdesc=dofdesc)

        abs_principal_curvatures = [
                abs(x) for x in _small_sym_mat_eigenvalues(shape_op)]
        from pymbolic.primitives import Max
        return cse(Max(tuple(abs_principal_curvatures)))
    else:
        raise NotImplementedError("curvature criterion not implemented in %d "
                "dimensions" % ambient_dim)


def _scaled_max_curvature(ambient_dim, dim=None, dofdesc=None):
    """An attempt at a unit-less, scale-invariant quantity that characterizes
    'how much curviness there is on an element'. Values seem to hover around 1
    on typical meshes. Empirical evidence suggests that elements exceeding
    a threshold of about 0.8-1 will have high QBX truncation error.
    """

    return (
        _max_curvature(ambient_dim, dim, dofdesc=dofdesc)
        * _mapping_max_stretch_factor(ambient_dim, dim=dim, dofdesc=dofdesc))

# }}}


# {{{ qbx-specific geometry

def _expansion_radii_factor(ambient_dim, dim):
    if dim is None:
        dim = ambient_dim - 1

    dim_fudge_factor = 0.5 if dim == 2 else 1.0
    return 0.5 * dim_fudge_factor


def _quad_resolution(ambient_dim, dim=None, granularity=None, dofdesc=None):
    """This measures the quadrature resolution across the
    mesh. In a 1D uniform mesh of uniform 'parametrization speed', it
    should be the same as the element length.

    In multiple dimensions (i.e. with multiple quadrature resolutions
    depending on direction), this measure returns the coarsest of these resolution,
    is invariant with respect to rotation of the global coordinate
    system, and invariant with respect to vertex ordering of the reference
    element.
    """

    from_dd = as_dofdesc(dofdesc)
    to_dd = from_dd.copy(granularity=granularity)

    stretch = _mapping_max_stretch_factor(ambient_dim, dim=dim, dofdesc=from_dd)
    return interpolate(stretch, from_dd, to_dd)


def _source_danger_zone_radii(ambient_dim, dim=None,
        granularity=None, dofdesc=None):
    # This should be the expression of the expansion radii, but
    #
    # - in reference to the stage 2 discretization
    # - multiplied by 0.75 because
    #
    #   - Setting this equal to the expansion radii ensures that *every*
    #     stage 2 element will be refined, which is wasteful.
    #     (so this needs to be smaller than that)
    #   - Setting this equal to half the expansion radius will not provide
    #     a refinement 'buffer layer' at a 2x coarsening fringe.

    return 0.75 * expansion_radii(ambient_dim,
            dim=dim, granularity=granularity, dofdesc=dofdesc)


def _close_target_tunnel_radii(ambient_dim, dim=None,
        granularity=None, dofdesc=None):
    return 0.5 * expansion_radii(ambient_dim,
            dim=dim, granularity=granularity, dofdesc=dofdesc)


def expansion_radii(ambient_dim, dim=None, granularity=None, dofdesc=None):
    factor = _expansion_radii_factor(ambient_dim, dim)
    return cse(factor * _quad_resolution(ambient_dim, dim=dim,
        granularity=granularity, dofdesc=dofdesc),
        "expansion_radii",
        cse_scope.DISCRETIZATION)


def expansion_centers(ambient_dim, side, dim=None, dofdesc=None):
    x = nodes(ambient_dim, dofdesc=dofdesc)
    normals = normal(ambient_dim, dim=dim, dofdesc=dofdesc)
    radii = expansion_radii(ambient_dim, dim=dim,
            granularity=GRANULARITY_NODE, dofdesc=dofdesc)

    centers = x + side * radii * normals
    return cse(centers.as_vector(),
            "expansion_centers",
            cse_scope.DISCRETIZATION)


def interleaved_expansion_centers(ambient_dim, dim=None, dofdesc=None):
    centers = (
            expansion_centers(ambient_dim, -1, dim=dim, dofdesc=dofdesc),
            expansion_centers(ambient_dim, +1, dim=dim, dofdesc=dofdesc)
            )

    source = as_dofdesc(dofdesc)
    target = source.copy(granularity=GRANULARITY_CENTER)
    return interpolate(centers, source, target)


def h_max(ambient_dim, dim=None, dofdesc=None):
    """Defines a maximum element size in the discretization."""

    r = _quad_resolution(ambient_dim, dim=dim, dofdesc=dofdesc)
    return cse(NodeMax(r),
            "h_max",
            cse_scope.DISCRETIZATION)


def h_min(ambient_dim, dim=None, dofdesc=None):
    """Yields an approximate minimum element size in the discretization."""

    r = _quad_resolution(ambient_dim, dim=dim, dofdesc=dofdesc)
    return cse(NodeMin(r),
            "h_min",
            cse_scope.DISCRETIZATION)


def weights_and_area_elements(ambient_dim, dim=None, dofdesc=None):
    """Combines :func:`area_element` and :class:`QWeight`."""
    dofdesc = as_dofdesc(dofdesc)
    area = area_element(ambient_dim, dim=dim, dofdesc=dofdesc)

    return cse(area * QWeight(dofdesc=dofdesc),
            "weights_area_elements",
            cse_scope.DISCRETIZATION)

# }}}


# {{{ operators

@expr_dataclass()
class Interpolation(Expression):
    """Interpolate quantity from a DOF described by *from_dd* to a DOF
    described by *to_dd*."

    .. autoattribute:: from_dd
    .. autoattribute:: to_dd
    .. autoattribute:: operand
    """

    from_dd: DOFDescriptor
    """A descriptor for the geometry on which *operand* is defined."""
    to_dd: DOFDescriptor
    """A descriptor for the geometry to which to interpolate *operand* to."""
    operand: Operand
    """An expression or array of expressions to interpolate. Arrays are
    interpolated componentwise.
    """

    def __new__(cls,
                from_dd: DOFDescriptorLike,
                to_dd: DOFDescriptorLike,
                operand: Operand) -> "Interpolation":
        from_dd = as_dofdesc(from_dd)
        to_dd = as_dofdesc(to_dd)

        if from_dd == to_dd:
            return operand  # type: ignore[return-value]

        if isinstance(operand, np.ndarray | MultiVector):
            warn(f"Passing {type(operand)} directly to {cls.__name__!r} "
                 "is deprecated and will result in an error from 2025. Use "
                 "the 'interpolate' function instead.",
                 DeprecationWarning, stacklevel=3)

            def make_op(operand_i):
                return cls(from_dd, to_dd, operand_i)

            return componentwise(make_op, operand)  # type: ignore[return-value]
        else:
            return Expression.__new__(cls)

    def __post_init__(self) -> None:
        if not isinstance(self.from_dd, DOFDescriptor):
            warn("Passing a 'from_dd' that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "from_dd", as_dofdesc(self.from_dd))

        if not isinstance(self.to_dd, DOFDescriptor):
            warn("Passing a 'to_dd' that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "to_dd", as_dofdesc(self.to_dd))


def interp(from_dd, to_dd, operand):
    warn("Calling 'interp' is deprecated and it will be removed in 2025. Use "
         "'interpolate' instead (has a different argument order).",
         DeprecationWarning, stacklevel=2)

    return Interpolation(as_dofdesc(from_dd), as_dofdesc(to_dd), operand)


@for_each_expression
def interpolate(operand: ArithmeticExpressionT,
                from_dd: DOFDescriptorLike,
                to_dd: DOFDescriptorLike) -> ArithmeticExpressionT | Interpolation:
    from_dd = as_dofdesc(from_dd)
    to_dd = as_dofdesc(to_dd)

    if from_dd == to_dd:
        return operand

    return Interpolation(from_dd, to_dd, operand)


@expr_dataclass()
class SingleScalarOperandExpression(Expression):
    """
    .. autoattribute:: operand
    """

    operand: Operand
    """An expression or an array on which to apply the operation."""

    def __new__(cls,
                operand: Operand | None = None) -> "SingleScalarOperandExpression":
        if isinstance(operand, np.ndarray | MultiVector):
            name = cls.mapper_method[4:]
            warn(f"Passing {type(operand)} directly to {cls.__name__!r} "
                 "is deprecated and will result in an error from 2025. Use "
                 f"the '{name}' function instead.",
                 DeprecationWarning, stacklevel=3)

            def make_op(operand_i):
                return cls(operand_i)

            return componentwise(make_op, operand)  # type: ignore[return-value]
        else:
            return Expression.__new__(cls)


@expr_dataclass()
class NodeSum(SingleScalarOperandExpression):
    """Bases: :class:`~pytential.symbolic.primitives.Expression`.

    Implements a global sum over all discretization nodes.
    """


@for_each_expression
def node_sum(expr: ArithmeticExpression) -> NodeSum:
    return NodeSum(expr)


@expr_dataclass()
class NodeMax(SingleScalarOperandExpression):
    """Bases: :class:`~pytential.symbolic.primitives.Expression`.

    Implements a global maximum over all discretization nodes.
    """


@for_each_expression
def node_max(expr: ArithmeticExpression) -> NodeMax:
    return NodeMax(expr)


@expr_dataclass()
class NodeMin(SingleScalarOperandExpression):
    """Bases: :class:`~pytential.symbolic.primitives.Expression`.

    Implements a global minimum over all discretization nodes.
    """


@for_each_expression
def node_min(expr: ArithmeticExpression) -> NodeMin:
    return NodeMin(expr)


def integral(ambient_dim, dim, operand, dofdesc=None):
    """A volume integral of *operand*."""

    dofdesc = as_dofdesc(dofdesc)
    return node_sum(
            area_element(ambient_dim, dim, dofdesc)
            * QWeight(dofdesc)
            * operand)


@expr_dataclass()
class SingleScalarOperandExpressionWithWhere(Expression):
    """
    .. autoattribute:: operand
    .. autoattribute:: dofdesc
    """

    operand: Operand
    """An expression or an array on which to apply the operation."""

    # pylint: disable-next=invalid-field-call
    dofdesc: DOFDescriptor = field(default_factory=lambda: DEFAULT_DOFDESC)
    """The descriptor for the geometry where the *operand* is defined."""

    def __new__(cls,
                operand: Operand | None = None,
                dofdesc: DOFDescriptorLike | None = None,
                ) -> "SingleScalarOperandExpressionWithWhere":
        if isinstance(operand, np.ndarray | MultiVector):
            name = cls.mapper_method[4:]
            warn(f"Passing {type(operand)} directly to {cls.__name__!r} "
                 "is deprecated and will result in an error from 2025. Use "
                 f"the '{name}' function instead.",
                 DeprecationWarning, stacklevel=2)

            def make_op(operand_i):
                return cls(operand_i, as_dofdesc(dofdesc))

            return componentwise(make_op, operand)  # type: ignore[return-value]
        else:
            return Expression.__new__(cls)

    def __post_init__(self) -> None:
        if not isinstance(self.dofdesc, DOFDescriptor):
            warn("Passing a 'dofdesc' that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "dofdesc", as_dofdesc(self.dofdesc))


@expr_dataclass()
class ElementwiseSum(SingleScalarOperandExpressionWithWhere):
    """Bases: :class:`~pytential.symbolic.primitives.Expression`.

    Returns a vector of DOFs with all entries on each element set
    to the sum of DOFs on that element.
    """


@for_each_expression
def elementwise_sum(expr: ArithmeticExpression,
                    dofdesc: DOFDescriptorLike = None) -> ElementwiseSum:
    return ElementwiseSum(expr, as_dofdesc(dofdesc))


@expr_dataclass()
class ElementwiseMin(SingleScalarOperandExpressionWithWhere):
    """Bases: :class:`~pytential.symbolic.primitives.Expression`.

    Returns a vector of DOFs with all entries on each element set
    to the minimum of DOFs on that element.
    """


@for_each_expression
def elementwise_min(expr: ArithmeticExpression,
                    dofdesc: DOFDescriptorLike = None) -> ElementwiseMin:
    return ElementwiseMin(expr, as_dofdesc(dofdesc))


@expr_dataclass()
class ElementwiseMax(SingleScalarOperandExpressionWithWhere):
    """Bases: :class:`~pytential.symbolic.primitives.Expression`.

    Returns a vector of DOFs with all entries on each element set
    to the maximum of DOFs on that element.
    """


@for_each_expression
def elementwise_max(expr: ArithmeticExpression,
                    dofdesc: DOFDescriptorLike = None) -> ElementwiseMax:
    return ElementwiseMax(expr, as_dofdesc(dofdesc))


@expr_dataclass()
class Ones(Expression):
    """A DOF-vector that is constant *one* on the whole discretization.
    """

    # pylint: disable-next=invalid-field-call
    dofdesc: DOFDescriptor = field(default_factory=lambda: DEFAULT_DOFDESC)
    """A descriptor for the discretization where the array is defined."""

    def __post_init__(self) -> None:
        if not isinstance(self.dofdesc, DOFDescriptor):
            warn("Passing a 'dofdesc' that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "dofdesc", as_dofdesc(self.dofdesc))


def ones_vec(dim, dofdesc=None):
    from pytools.obj_array import make_obj_array

    dofdesc = as_dofdesc(dofdesc)
    return MultiVector(make_obj_array(dim*[Ones(dofdesc)]))


def area(ambient_dim, dim, dofdesc=None):
    dofdesc = as_dofdesc(dofdesc)
    return cse(integral(ambient_dim, dim, Ones(dofdesc), dofdesc), "area",
            cse_scope.DISCRETIZATION)


def mean(ambient_dim, dim, operand, dofdesc=None):
    return (
            integral(ambient_dim, dim, operand, dofdesc)
            / area(ambient_dim, dim, dofdesc))


@expr_dataclass()
class IterativeInverse(Expression):
    """A symbolic :math:`A x = b` linear solve expression.

    .. autoattribute:: expression
    .. autoattribute:: rhs
    .. autoattribute:: variable_name
    .. autoattribute:: extra_vars
    .. autoattribute:: dofdesc
    """

    expression: ArithmeticExpression
    """The operator *A* used in the linear solve."""
    rhs: ArithmeticExpression
    """The right-hand side variable used in the linear solve."""
    variable_name: str
    """The name of the variable to solve for."""

    # pylint: disable-next=invalid-field-call
    extra_vars: dict[str, Variable] = field(default_factory=dict)
    """A dictionary of additional variables required to define the operator."""

    # pylint: disable-next=invalid-field-call
    dofdesc: DOFDescriptor = field(default_factory=lambda: DEFAULT_DOFDESC)
    """A descriptor for the geometry on which the solution is defined."""

    def __post_init__(self) -> None:
        if not isinstance(self.dofdesc, DOFDescriptor):
            warn("Passing a 'dofdesc' that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "dofdesc", as_dofdesc(self.dofdesc))


class Derivative(DerivativeBase):
    """A symbolic derivative.

    This mechanism cannot be used to take more than one derivative at a time.

    .. automethod:: __call__
    .. automethod:: dnabla
    .. automethod:: resolve
    """

    @property
    def nabla(self):
        raise ValueError("Derivative.nabla should not be used"
                "--use Derivative.dnabla instead. (Note the extra 'd')"
                "To explain: 'nabla' was intended to be "
                "dimension-independent, which turned out to be a bad "
                "idea.")

    @staticmethod
    def resolve(expr):
        from pytential.symbolic.mappers import DerivativeBinder
        return DerivativeBinder()(expr)


def dd_axis(axis, ambient_dim, operand):
    """Return the derivative along (XYZ) axis *axis*
    (in *ambient_dim*-dimensional space) of *operand*.
    """
    from pytools.obj_array import obj_array_vectorize
    if isinstance(operand, np.ndarray) and operand.dtype.char == "O":
        def dd_axis_comp(operand_i):
            return dd_axis(axis, ambient_dim, operand_i)

        return obj_array_vectorize(dd_axis_comp, operand)

    d = Derivative()

    unit_vector = np.zeros(ambient_dim)
    unit_vector[axis] = 1

    unit_mvector = MultiVector(unit_vector)

    return d.resolve(
            (unit_mvector.scalar_product(d.dnabla(ambient_dim)))
            * d(operand))


d_dx = partial(dd_axis, 0)
d_dy = partial(dd_axis, 1)
d_dz = partial(dd_axis, 2)


def grad_mv(ambient_dim, operand):
    """Return the *ambient_dim*-dimensional gradient of
    *operand* as a :class:`pymbolic.geometric_algebra.MultiVector`.
    """

    d = Derivative()
    return d.resolve(
            d.dnabla(ambient_dim) * d(operand))


def grad(ambient_dim, operand):
    """Return the *ambient_dim*-dimensional gradient of
    *operand* as a :class:`numpy.ndarray`.
    """
    return grad_mv(ambient_dim, operand).as_vector()


def laplace(ambient_dim, operand):
    d = Derivative()
    nabla = d.dnabla(ambient_dim)
    return d.resolve(nabla | d(
        d.resolve(nabla * d(operand)))).as_scalar()


# {{{ potentials

def hashable_kernel_args(kernel_arguments):
    hashable_args = []
    for key, val in sorted(kernel_arguments.items()):
        if isinstance(val, np.ndarray):
            val = tuple(val)
        hashable_args.append((key, val))

    return tuple(hashable_args)


@expr_dataclass(init=False, hash=False)
class IntG(Expression):
    r"""
    .. math::

        \int_\Gamma T \left[\sum S_k[G](x-y) \sigma_k(y)\right] \,\mathrm{d} S_y

    where :math:`\sigma_k` is the k-th *density*, :math:`G` is a Green's
    function, :math:`S_k` are source derivative operators and :math:`T` is a
    target derivative operator.

    .. autoattribute:: target_kernel
    .. autoattribute:: source_kernels
    .. autoattribute:: densities
    .. autoattribute:: qbx_forced_limit
    .. autoattribute:: source
    .. autoattribute:: target
    .. autoattribute:: kernel_arguments
    """

    target_kernel: Kernel
    """An instance of :class:`~sumpy.kernel.Kernel` with only target dervatives
    attached. This represents the target derivative operator :math:`T` above.

    Note that the term ``target_kernel`` is bad as it's not a kernel and merely
    represents a target derivative operator. This name will change once :mod:`sumpy`
    properly supports derivative operators. This also means that the user has to
    make sure that base kernels of all the kernels passed are the same.
    """
    source_kernels: tuple[Kernel, ...]
    """A tuple of instances of :class:`~sumpy.kernel.Kernel` with only source
    derivatives attached. k-th elements represents the k-th source derivative
    operator above.
    """
    densities: tuple[Expression, ...]
    """A tuple of density expressions. Length of this tuple must match the length
    of the *source_kernels* arguments.
    """
    qbx_forced_limit: QBXForcedLimit
    """Limit used for the QBX expansions. Can take one of the values

    * *None*: may be used to avoid expressing a side preference for close
      evaluation.
    * ``+1``: if the output is required to originate from a QBX center on the "+"
      side of the boundary.
    * ``-1``: for the "-" side.
    * ``'avg'``: may be used as a shorthand to evaluate this potential as an
      average of the ``+1`` and the ``-1`` value.
    * ``+2`` may be used to *allow* evaluation QBX center on the "+" side of the
      (but disallow evaluation using a center on the "-" side).
    * ``-2``: for the "-" side.

    Evaluation at a target with a value of ``±1`` in *qbx_forced_limit* will
    fail if no QBX center is found. To allow potential evaluation at the target
    to succeeds even if no applicable QBX center is found use ``±2``.
    """

    # pylint: disable-next=invalid-field-call
    source: DOFDescriptor = field(default_factory=lambda: DEFAULT_DOFDESC)
    """The symbolic name of the source discretization. This name is bound to a
    concrete :class:`~pytential.source.LayerPotentialSourceBase`
    by :func:`pytential.bind`.
    """

    # pylint: disable-next=invalid-field-call
    target: DOFDescriptor = field(default_factory=lambda: DEFAULT_DOFDESC)
    """The symbolic name of the set of targets. This name gets assigned to a
    concrete target set by :func:`pytential.bind`.
    """

    # pylint: disable-next=invalid-field-call
    kernel_arguments: dict[str, Operand] = field(default_factory=dict)
    """A dictionary mapping named :class:`~sumpy.kernel.Kernel` arguments
    (see :meth:`~sumpy.kernel.Kernel.get_args` and
    :meth:`~sumpy.kernel.Kernel.get_source_args`) to expressions that determine
    them.
    """

    def __init__(
            self,
            target_kernel: Kernel,
            source_kernels: Iterable[Kernel],
            densities: Iterable[Expression],
            qbx_forced_limit: QBXForcedLimit,
            source: DOFDescriptorLike | None = None,
            target: DOFDescriptorLike | None = None,
            kernel_arguments: dict[str, Any] | None = None,
            **kwargs: Any
            ) -> None:
        if kernel_arguments is None:
            kernel_arguments = {}

        if kwargs:
            warn(f"Passing named '**kwargs' to {type(self).__name__!r} is "
                 "deprecated and will result in an error in 2025. Use the "
                 "'kernel_arguments' argument instead.",
                 DeprecationWarning, stacklevel=2)

            kernel_arguments = kernel_arguments.copy()
            for name, value in kwargs.items():
                if name in kernel_arguments:
                    raise ValueError(f"'{name}' already set in 'kernel_arguments'")

                kernel_arguments[name] = value

        object.__setattr__(self, "target_kernel", target_kernel)
        object.__setattr__(self, "source_kernels", source_kernels)
        object.__setattr__(self, "densities", densities)
        object.__setattr__(self, "qbx_forced_limit", qbx_forced_limit)
        object.__setattr__(self, "source", source)
        object.__setattr__(self, "target", target)
        object.__setattr__(self, "kernel_arguments", kernel_arguments)

        self.__post_init__()

    def __post_init__(self) -> None:
        if self.qbx_forced_limit not in {-1, +1, -2, +2, "avg", None}:
            raise ValueError(
                f"Invalid value for 'qbx_forced_limit': {self.qbx_forced_limit}"
            )

        if not isinstance(self.source_kernels, tuple):
            warn(f"'source_kernels' is not tuple ({type(self.source_kernels)}). "
                 "Passing a different type is deprecated and will stop working in "
                 "2025.", DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "source_kernels", tuple(self.source_kernels))

        if not isinstance(self.densities, tuple):
            warn(f"'densities' is not tuple ({type(self.densities)}). "
                 "Passing a different type is deprecated and will stop working in "
                 "2025.", DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "densities", tuple(self.densities))

        if not isinstance(self.source, DOFDescriptor):
            warn("Passing a 'source' descriptor that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "source", as_dofdesc(self.source))

        if not isinstance(self.target, DOFDescriptor):
            warn("Passing a 'target' descriptor that is not a 'DOFDescriptor' to "
                 f"{type(self).__name__!r} is deprecated and will stop working "
                 "in 2025. Use 'as_dofdesc' to convert the descriptor.",
                 DeprecationWarning, stacklevel=2)

            object.__setattr__(self, "target", as_dofdesc(self.target))

        if not isinstance(self.kernel_arguments, dict):
            warn(f"'kernel_arguments' is not a dict ({type(self.kernel_arguments)}). "
                 "Passing a different type is deprecated and will stop being "
                 "supported in 2025.", DeprecationWarning, stacklevel=2)

            kernel_arguments = self.kernel_arguments if self.kernel_arguments else {}
            object.__setattr__(self, "kernel_arguments", dict(kernel_arguments))

        from pytools import single_valued

        kernels = (*self.source_kernels, self.target_kernel)
        single_valued(kernel.get_base_kernel() for kernel in kernels)

        kernel_arg_names = set()
        for kernel in kernels:
            for karg in (kernel.get_args() + kernel.get_source_args()):
                kernel_arg_names.add(karg.loopy_arg.name)

        provided_arg_names = set(self.kernel_arguments.keys())  # pylint: disable=no-member
        missing_args = kernel_arg_names - provided_arg_names
        if missing_args:
            raise ValueError(
                "Kernel argument(s) not supplied: '{}'".format(", ".join(missing_args))
            )

        # FIXME: this check is clearly wrong :(
        extra_args = provided_arg_names - kernel_arg_names
        if missing_args:
            raise ValueError(
                "Kernel argument(s) not recognized: '{}'".format(", ".join(extra_args))
            )

    def copy(self, **kwargs) -> "IntG":
        warn(f"'{type(self).__name__}.copy' is deprecated and will be removed in "
             f"2025. {type(self)} is a dataclass now and can use "
             "'dataclasses.replace'.", DeprecationWarning, stacklevel=2)

        from dataclasses import replace
        return replace(self, **kwargs)

    def __eq__(self, other: Any) -> bool:
        if self is other:
            return True
        if self.__class__ is not other.__class__:
            return False
        if hash(self) != hash(other):
            return False

        return (
            self.__class__ == other.__class__
            and self.target_kernel == other.target_kernel
            and self.source_kernels == other.source_kernels
            and self.densities == other.densities
            and self.source == other.source
            and self.target == other.target
            and (
                hashable_kernel_args(self.kernel_arguments)
                == hashable_kernel_args(other.kernel_arguments))
            )

    def __hash__(self) -> int:
        return hash((
            self.target_kernel,
            self.source_kernels,
            self.densities,
            self.qbx_forced_limit,
            self.source, self.target,
            hashable_kernel_args(self.kernel_arguments)
            ))


_DIR_VEC_NAME = "dsource_vec"


def _insert_source_derivative_into_kernel(kernel):
    # Inserts the source derivative at the innermost
    # kernel wrapping level.
    from sumpy.kernel import DirectionalSourceDerivative

    if kernel.get_base_kernel() is kernel:
        return DirectionalSourceDerivative(
                kernel, dir_vec_name=_DIR_VEC_NAME)
    else:
        return kernel.replace_inner_kernel(
                _insert_source_derivative_into_kernel(kernel.inner_kernel))


def _get_dir_vec(dsource, ambient_dim):
    from pymbolic.mapper.coefficient import (
            CoefficientCollector as CoefficientCollectorBase)

    class _DSourceCoefficientFinder(CoefficientCollectorBase):
        def map_nabla_component(self, expr):
            return {expr: 1}

        def map_variable(self, expr):
            return {1: expr}

        def map_common_subexpression(self, expr):
            return {1: expr}

        def map_quotient(self, expr):
            return {1: expr}

    coeffs = _DSourceCoefficientFinder()(dsource)

    dir_vec = np.zeros(ambient_dim, object)
    for i in range(ambient_dim):
        dir_vec[i] = coeffs.pop(NablaComponent(i, None), 0)

    if coeffs:
        raise RuntimeError("source derivative expression contained constant term")

    return dir_vec


def int_g_dsource(ambient_dim, dsource, kernel, density,
            qbx_forced_limit, source=None, target=None,
            kernel_arguments=None, **kwargs):
    r"""
    .. math::

        \int_\Gamma \operatorname{dsource} \dot \nabla_y G(x-y) \sigma(y) dS_y

    where :math:`\sigma` is *density* and *dsource* is a multivector.
    Note that the first product in the integrand is a geometric product.
    """

    if kernel_arguments is None:
        kernel_arguments = {}

    if isinstance(kernel_arguments, tuple):
        kernel_arguments = dict(kernel_arguments)

    kernel = _insert_source_derivative_into_kernel(kernel)

    from pytools.obj_array import make_obj_array
    nabla = MultiVector(make_obj_array(
        [NablaComponent(axis, None)
            for axis in range(ambient_dim)]))

    def add_dir_vec_to_kernel_args(coeff):
        result = kernel_arguments.copy()
        result[_DIR_VEC_NAME] = _get_dir_vec(coeff, ambient_dim)
        return result

    density = cse(density)
    return (dsource*nabla).map(
            lambda coeff: int_g_vec(
                kernel,
                density, qbx_forced_limit, source, target,
                kernel_arguments=add_dir_vec_to_kernel_args(coeff),
                **kwargs))

# }}}


# {{{ non-dimension-specific operators

# {{{ geometric calculus


class _unspecified:  # noqa: N801
    pass


def int_g_vec(kernel, density, qbx_forced_limit, source=None, target=None,
        kernel_arguments=None, **kwargs):
    """
    Creates a vector of :class:`IntG` objects from one kernel with source and
    target derivatives and maps a vector of densities into a vector of
    :class:`IntG` objects.

    Historically :class:`IntG` objects supported only one source kernel and
    allowed multiple densities to get a vector of objects as a convenience
    function. Now that :class:`IntG` objects supports multiple source kernels
    with one density associated to each source kernel, the previous interface
    would lead to ambiguity. This function is intended to preserve the
    "vectorizing" behavior of of the constructor of :class:`IntG`
    for use cases where that is preferred.
    """
    from sumpy.kernel import (SourceTransformationRemover,
            TargetTransformationRemover)
    sxr = SourceTransformationRemover()
    txr = TargetTransformationRemover()

    target_kernel = sxr(kernel)
    source_kernels = (txr(kernel),)

    if kernel_arguments is None:
        kernel_arguments = {}

    if kwargs is not None:
        kernel_arguments = {**kernel_arguments, **kwargs}

    def make_op(operand_i):
        return IntG(
            target_kernel=target_kernel,
            source_kernels=source_kernels,
            densities=(operand_i,),
            qbx_forced_limit=qbx_forced_limit,
            source=as_dofdesc(source),
            target=as_dofdesc(target),
            kernel_arguments=kernel_arguments)

    if isinstance(density, np.ndarray | MultiVector):
        return componentwise(make_op, density)
    else:
        return make_op(density)


def S(kernel, density,
        qbx_forced_limit=_unspecified, source=None, target=None,
        kernel_arguments=None, **kwargs):

    if qbx_forced_limit is _unspecified:
        warn("not specifying qbx_forced_limit on call to 'S' is deprecated, "
                "defaulting to +1", stacklevel=2)
        qbx_forced_limit = +1

    return int_g_vec(kernel, density, qbx_forced_limit, source, target,
            kernel_arguments, **kwargs)


def tangential_derivative(ambient_dim, operand, dim=None, dofdesc=None):
    pder = (
            pseudoscalar(ambient_dim, dim, dofdesc)
            / area_element(ambient_dim, dim, dofdesc))

    # FIXME: Should be formula (3.25) in Dorst et al.
    d = Derivative()
    return d.resolve(
            (d.dnabla(ambient_dim) * d(operand)) >> pder)


def normal_derivative(ambient_dim, operand, dim=None, dofdesc=None):
    d = Derivative()
    return d.resolve(
            (normal(ambient_dim, dim, dofdesc).scalar_product(d.dnabla(ambient_dim)))
            * d(operand))


def normal_second_derivative(ambient_dim, operand, dim=None, dofdesc=None):
    d = Derivative()
    n = normal(ambient_dim, dim, dofdesc)
    nabla = d.dnabla(ambient_dim)

    return d.resolve(n.scalar_product(
        n.scalar_product(nabla) * d(d.resolve(nabla * d(operand)))
        ))


def Sp(kernel, *args, **kwargs):
    dofdesc = kwargs.get("target")
    if "qbx_forced_limit" not in kwargs:
        warn("not specifying qbx_forced_limit on call to 'Sp' is deprecated, "
                "defaulting to 'avg'", DeprecationWarning, stacklevel=2)
        kwargs["qbx_forced_limit"] = "avg"

    ambient_dim = kwargs.get("ambient_dim")
    if ambient_dim is None and isinstance(kernel, Kernel):
        ambient_dim = kernel.dim
    if ambient_dim is None:
        raise ValueError("ambient_dim must be specified, either through "
                "the kernel, or directly")
    dim = kwargs.pop("dim", None)

    return normal_derivative(
            ambient_dim,
            S(kernel, *args, **kwargs),
            dim=dim, dofdesc=dofdesc)


def Spp(kernel, *args, **kwargs):
    ambient_dim = kwargs.get("ambient_dim")
    if ambient_dim is None and isinstance(kernel, Kernel):
        ambient_dim = kernel.dim
    if ambient_dim is None:
        raise ValueError("ambient_dim must be specified, either through "
                "the kernel, or directly")
    dim = kwargs.pop("dim", None)

    dofdesc = kwargs.get("target")
    return normal_second_derivative(
            ambient_dim,
            S(kernel, *args, **kwargs),
            dim=dim, dofdesc=dofdesc)


def D(kernel, *args, **kwargs):
    ambient_dim = kwargs.get("ambient_dim")
    if ambient_dim is None and isinstance(kernel, Kernel):
        ambient_dim = kernel.dim
    if ambient_dim is None:
        raise ValueError("ambient_dim must be specified, either through "
                "the kernel, or directly")
    dim = kwargs.pop("dim", None)

    dofdesc = kwargs.get("source")

    if "qbx_forced_limit" not in kwargs:
        warn("not specifying qbx_forced_limit on call to 'D' is deprecated, "
                "defaulting to 'avg'", DeprecationWarning, stacklevel=2)
        kwargs["qbx_forced_limit"] = "avg"

    return int_g_dsource(
            ambient_dim,
            normal(ambient_dim, dim, dofdesc),
            kernel, *args, **kwargs).xproject(0)


def Dp(kernel, *args, **kwargs):
    ambient_dim = kwargs.get("ambient_dim")
    if ambient_dim is None and isinstance(kernel, Kernel):
        ambient_dim = kernel.dim
    if ambient_dim is None:
        raise ValueError("ambient_dim must be specified, either through "
                "the kernel, or directly")
    dim = kwargs.pop("dim", None)
    target = kwargs.get("target")
    if "qbx_forced_limit" not in kwargs:
        warn("not specifying qbx_forced_limit on call to 'Dp' is deprecated, "
                "defaulting to +1", DeprecationWarning, stacklevel=2)
        kwargs["qbx_forced_limit"] = +1
    return normal_derivative(
            ambient_dim,
            D(kernel, *args, **kwargs),
            dim=dim, dofdesc=target)

# }}}

# }}}

# }}}


# {{{ conventional vector calculus

def tangential_onb(ambient_dim, dim=None, dofdesc=None):
    """Return a matrix of shape ``(ambient_dim, dim)`` with orthogonal columns
    spanning the tangential space of the surface of *dofdesc*.
    """

    if dim is None:
        dim = ambient_dim - 1

    pd_mat = parametrization_derivative_matrix(ambient_dim, dim, dofdesc)

    # {{{ Gram-Schmidt

    orth_pd_mat = np.zeros_like(pd_mat)
    for k in range(pd_mat.shape[1]):
        avec = pd_mat[:, k]
        q = avec
        for j in range(k):
            q = q - np.dot(avec, orth_pd_mat[:, j])*orth_pd_mat[:, j]
        q = cse(q, f"q{k}")

        orth_pd_mat[:, k] = cse(q/sqrt(np.sum(q**2)), f"orth_pd_vec{k}_")

    # }}}

    return orth_pd_mat


def xyz_to_tangential(xyz_vec, dofdesc=None):
    ambient_dim = len(xyz_vec)
    tonb = tangential_onb(ambient_dim, dofdesc=dofdesc)
    return make_obj_array([
        np.dot(tonb[:, i], xyz_vec)
        for i in range(ambient_dim - 1)
        ])


def tangential_to_xyz(tangential_vec, dofdesc=None):
    ambient_dim = len(tangential_vec) + 1
    tonb = tangential_onb(ambient_dim, dofdesc=dofdesc)
    return sum(
        tonb[:, i] * tangential_vec[i]
        for i in range(ambient_dim - 1))


def project_to_tangential(xyz_vec, dofdesc=None):
    return tangential_to_xyz(
            cse(xyz_to_tangential(xyz_vec, dofdesc)),
            dofdesc)


def n_dot(vec, dofdesc=None):
    nrm = normal(len(vec), dofdesc=dofdesc).as_vector()

    return np.dot(nrm, vec)


def cross(vec_a, vec_b):
    assert len(vec_a) == len(vec_b) == 3

    from pytools import levi_civita
    from pytools.obj_array import make_obj_array
    return make_obj_array([
        sum(
            levi_civita((i, j, k)) * vec_a[j] * vec_b[k]
            for j in range(3) for k in range(3))
        for i in range(3)])


def n_cross(vec, dofdesc=None):
    return cross(normal(3, dofdesc=dofdesc).as_vector(), vec)


def div(vec):
    ambient_dim = len(vec)
    return sum(
            dd_axis(iaxis, ambient_dim, vec[iaxis])
            for iaxis in range(ambient_dim))


def curl(vec):
    from pytools import levi_civita
    from pytools.obj_array import make_obj_array

    return make_obj_array([
        sum(
            levi_civita((ell, m, n)) * dd_axis(m, 3, vec[n])
            for m in range(3) for n in range(3))
        for ell in range(3)])

# }}}


def pretty(expr):
    # Doesn't quite belong here, but this is exposed to the user as
    # "pytential.sym", so in here it goes.

    from pytential.symbolic.mappers import PrettyStringifyMapper
    stringify_mapper = PrettyStringifyMapper()
    from pymbolic.mapper.stringifier import PREC_NONE
    result = stringify_mapper(expr, PREC_NONE)

    splitter = "="*75 + "\n"

    cse_strs = stringify_mapper.get_cse_strings()
    if cse_strs:
        result = "\n".join(cse_strs)+"\n"+splitter+result

    return result

# vim: foldmethod=marker
