"""Scan primitive."""

from __future__ import division

__copyright__ = """
Copyright 2011-2012 Andreas Kloeckner
Copyright 2008-2011 NVIDIA Corporation
"""

__license__ = """
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Derived from thrust/detail/backend/cuda/detail/fast_scan.inl
within the Thrust project, https://code.google.com/p/thrust/

Direct browse link:
https://code.google.com/p/thrust/source/browse/thrust/detail/backend/cuda/detail/fast_scan.inl
"""



import numpy as np

import pyopencl as cl
import pyopencl.array as cl_array
from pyopencl.tools import dtype_to_ctype, bitlog2
import pyopencl._mymako as mako
from pyopencl._cluda import CLUDA_PREAMBLE
from pyopencl.tools import context_dependent_memoize





# {{{ preamble

SHARED_PREAMBLE = CLUDA_PREAMBLE + """//CL//
#define WG_SIZE ${wg_size}

/* SCAN_EXPR has no right know the indices it is scanning at because
each index may occur an undetermined number of times in the scan tree,
and thus index-based side computations cannot be meaningful. */

#define SCAN_EXPR(a, b) ${scan_expr}
#define INPUT_EXPR(i) (${input_expr})
%if is_segmented:
    #define IS_SEG_START(i, a) (${is_segment_start_expr})
%endif

${preamble}

typedef ${scan_ctype} scan_type;
typedef ${index_ctype} index_type;

// NO_SEG_BOUNDARY is the largest representable integer in index_type.
// This assumption is used in code below.
#define NO_SEG_BOUNDARY ${index_type_max}
"""

# }}}

# {{{ main scan code

# Algorithm: Each work group is responsible for one contiguous
# 'interval'. There are just enough intervals to fill all compute
# units.  Intervals are split into 'units'. A unit is what gets
# worked on in parallel by one work group.
#
# in index space:
# interval > unit > local-parallel > k-group
#
# (Note that there is also a transpose in here: The data is read
# with local ids along linear index order.)
#
# Each unit has two axes--the local-id axis and the k axis.
#
# unit 0:
# | | | | | | | | | | ----> lid
# | | | | | | | | | |
# | | | | | | | | | |
# | | | | | | | | | |
# | | | | | | | | | |
#
# |
# v k (fastest-moving in linear index)
#
# unit 1:
# | | | | | | | | | | ----> lid
# | | | | | | | | | |
# | | | | | | | | | |
# | | | | | | | | | |
# | | | | | | | | | |
#
# |
# v k (fastest-moving in linear index)
#
# ...
#
# At a device-global level, this is a three-phase algorithm, in
# which first each interval does its local scan, then a scan
# across intervals exchanges data globally, and the final update
# adds the exchanged sums to each interval.
#
# Exclusive scan is realized by allowing look-behind (access to the
# preceding item) in the final update, by means of a local shift.
#
# NOTE: All segment_start_in_X indices are relative to the start
# of the array.

SCAN_INTERVALS_SOURCE = SHARED_PREAMBLE + r"""//CL//

#define K ${k_group_size}

KERNEL
REQD_WG_SIZE(WG_SIZE, 1, 1)
void ${name_prefix}_scan_intervals(
    ${argument_signature},
    GLOBAL_MEM scan_type *partial_scan_buffer,
    const index_type N,
    const index_type interval_size
    %if is_first_level:
        , GLOBAL_MEM scan_type *interval_results
    %endif
    %if is_segmented and is_first_level:
        // NO_SEG_BOUNDARY if no segment boundary in interval.
        , GLOBAL_MEM index_type *g_first_segment_start_in_interval
    %endif
    %if store_segment_start_flags:
        , GLOBAL_MEM char *g_segment_start_flags
    %endif
    )
{
    // padded in WG_SIZE to avoid bank conflicts
    // index K in first dimension used for carry storage
    LOCAL_MEM scan_type ldata[K + 1][WG_SIZE + 1];

    %if is_segmented:
        LOCAL_MEM char l_segment_start_flags[K][WG_SIZE];
        LOCAL_MEM index_type l_first_segment_start_in_subtree[WG_SIZE];

        // only relevant/populated for local id 0
        index_type first_segment_start_in_interval = NO_SEG_BOUNDARY;

        index_type first_segment_start_in_k_group, first_segment_start_in_subtree;
    %endif

    // {{{ set up local data for input_fetch_exprs if any of them are stenciled

    <%
        fetch_expr_offsets = {}
        for name, arg_name, ife_offset in input_fetch_exprs:
            fetch_expr_offsets.setdefault(arg_name, set()).add(ife_offset)

        local_fetch_expr_args = set(
            arg_name
            for arg_name, ife_offsets in fetch_expr_offsets.items()
            if -1 in ife_offsets or len(ife_offsets) > 1)
    %>

    %for arg_name in local_fetch_expr_args:
        LOCAL_MEM ${arg_ctypes[arg_name]} l_${arg_name}[WG_SIZE*K];
    %endfor

    // }}}

    const index_type interval_begin = interval_size * GID_0;
    const index_type interval_end   = min(interval_begin + interval_size, N);

    const index_type unit_size  = K * WG_SIZE;

    index_type unit_base = interval_begin;

    %for is_tail in [False, True]:

        %if not is_tail:
            for(; unit_base + unit_size <= interval_end; unit_base += unit_size)
        %else:
            if (unit_base < interval_end)
        %endif

        {

            // {{{ carry out input_fetch_exprs
            // (if there are ones that need to be fetched into local)

            %if local_fetch_expr_args:
                for(index_type k = 0; k < K; k++)
                {
                    const index_type offset = k*WG_SIZE + LID_0;
                    const index_type read_i = unit_base + offset;

                    %for arg_name in local_fetch_expr_args:
                        %if is_tail:
                        if (read_i < interval_end)
                        %endif
                        {
                            l_${arg_name}[offset] = ${arg_name}[read_i];
                        }
                    %endfor
                }

                local_barrier();
            %endif

            // }}}

            // {{{ read a unit's worth of data from global

            for(index_type k = 0; k < K; k++)
            {
                const index_type offset = k*WG_SIZE + LID_0;
                const index_type read_i = unit_base + offset;

                %if is_tail:
                if (read_i < interval_end)
                %endif
                {
                    %for name, arg_name, ife_offset in input_fetch_exprs:
                        ${arg_ctypes[arg_name]} ${name};

                        %if arg_name in local_fetch_expr_args:
                            if (offset + ${ife_offset} >= 0)
                                ${name} = l_${arg_name}[offset + ${ife_offset}];
                            else if (read_i + ${ife_offset} >= 0)
                                ${name} = ${arg_name}[read_i + ${ife_offset}];
                            /*
                            else
                                if out of bounds, name is left undefined */

                        %else:
                            // ${arg_name} gets fetched directly from global
                            ${name} = ${arg_name}[read_i];

                        %endif
                    %endfor

                    scan_type scan_value = INPUT_EXPR(read_i);

                    const index_type o_mod_k = offset % K;
                    const index_type o_div_k = offset / K;
                    ldata[o_mod_k][offset / K] = scan_value;

                    %if is_segmented:
                        bool is_seg_start = IS_SEG_START(read_i, scan_value);
                        l_segment_start_flags[o_mod_k][o_div_k] = is_seg_start;
                    %endif
                    %if store_segment_start_flags:
                        g_segment_start_flags[read_i] = is_seg_start;
                    %endif
                }
            }

            // }}}

            // {{{ carry in from previous unit, if applicable

            %if is_segmented:
                local_barrier();

                first_segment_start_in_k_group = NO_SEG_BOUNDARY;
                if (l_segment_start_flags[0][LID_0])
                    first_segment_start_in_k_group = unit_base + K*LID_0;
            %endif

            if (LID_0 == 0 && unit_base != interval_begin
                %if is_segmented:
                    && !l_segment_start_flags[0][0]
                %endif
                )
            {
                ldata[0][0] = SCAN_EXPR(ldata[K][WG_SIZE - 1], ldata[0][0]);
            }

            // }}}

            local_barrier();

            // {{{ scan along k (sequentially in each work item)

            scan_type sum = ldata[0][LID_0];

            %if is_tail:
                const index_type offset_end = interval_end - unit_base;
            %endif

            for(index_type k = 1; k < K; k++)
            {
                %if is_tail:
                if (K * LID_0 + k < offset_end)
                %endif
                {
                    scan_type tmp = ldata[k][LID_0];
                    index_type seq_i = unit_base + K*LID_0 + k;

                    %if is_segmented:
                    if (l_segment_start_flags[k][LID_0])
                    {
                        first_segment_start_in_k_group = min(
                            first_segment_start_in_k_group,
                            seq_i);
                        sum = tmp;
                    }
                    else
                    %endif
                        sum = SCAN_EXPR(sum, tmp);

                    ldata[k][LID_0] = sum;
                }
            }

            // }}}

            // store carry in out-of-bounds (padding) array entry (index K) in the K direction
            ldata[K][LID_0] = sum;

            %if is_segmented:
                l_first_segment_start_in_subtree[LID_0] = first_segment_start_in_k_group;
            %endif

            local_barrier();

            // {{{ tree-based local parallel scan

            // This tree-based scan works as follows:
            // - Each work item adds the previous item to its current state
            // - barrier
            // - Each work item adds in the item from two positions to the left
            // - barrier
            // - Each work item adds in the item from four positions to the left
            // ...
            // At the end, each item has summed all prior items.

            // across k groups, along local id
            // (uses out-of-bounds k=K array entry for storage)

            scan_type val = ldata[K][LID_0];

            <% scan_offset = 1 %>

            % while scan_offset <= wg_size:
                // {{{ reads from local allowed, writes to local not allowed

                if (LID_0 >= ${scan_offset})
                {
                    scan_type tmp = ldata[K][LID_0 - ${scan_offset}];
                    % if is_tail:
                    if (K*LID_0 < offset_end)
                    % endif
                    {
                        %if is_segmented:
                            if (l_first_segment_start_in_subtree[LID_0] == NO_SEG_BOUNDARY)
                                val = SCAN_EXPR(tmp, val);
                        %else:
                            val = SCAN_EXPR(tmp, val);
                        %endif
                    }

                    %if is_segmented:
                        // Prepare for l_first_segment_start_in_subtree, below.

                        // Note that this update must take place *even* if we're
                        // out of bounds.

                        first_segment_start_in_subtree = min(
                            l_first_segment_start_in_subtree[LID_0],
                            l_first_segment_start_in_subtree[LID_0 - ${scan_offset}]);
                    %endif
                }
                %if is_segmented:
                    else
                    {
                        first_segment_start_in_subtree =
                            l_first_segment_start_in_subtree[LID_0];
                    }
                %endif

                // }}}

                local_barrier();

                // {{{ writes to local allowed, reads from local not allowed

                ldata[K][LID_0] = val;
                %if is_segmented:
                    l_first_segment_start_in_subtree[LID_0] =
                        first_segment_start_in_subtree;
                %endif

                // }}}

                local_barrier();

                %if 0:
                if (LID_0 == 0)
                {
                    printf("${scan_offset}: ");
                    for (int i = 0; i < WG_SIZE; ++i)
                    {
                        if (l_first_segment_start_in_subtree[i] == NO_SEG_BOUNDARY)
                            printf("- ");
                        else
                            printf("%d ", l_first_segment_start_in_subtree[i]);
                    }
                    printf("\n");
                }
                %endif

                <% scan_offset *= 2 %>
            % endwhile

            // }}}

            // {{{ update local values

            if (LID_0 > 0)
            {
                sum = ldata[K][LID_0 - 1];

                for(index_type k = 0; k < K; k++)
                {
                    bool do_update = true;
                    %if is_tail:
                        do_update = K * LID_0 + k < offset_end;
                    %endif
                    %if is_segmented:
                        do_update = unit_base + K * LID_0 + k
                            < first_segment_start_in_k_group;
                    %endif

                    if (do_update)
                    {
                        scan_type tmp = ldata[k][LID_0];
                        ldata[k][LID_0] = SCAN_EXPR(sum, tmp);
                    }
                }
            }

            %if is_segmented:
                if (LID_0 == 0)
                {
                    // update interval-wide first-seg variable from current unit
                    first_segment_start_in_interval = min(
                        first_segment_start_in_interval,
                        l_first_segment_start_in_subtree[WG_SIZE-1]);
                }
            %endif

            // }}}

            local_barrier();

            // {{{ write data

            for (index_type k = 0; k < K; k++)
            {
                const index_type offset = k*WG_SIZE + LID_0;

                %if is_tail:
                if (unit_base + offset < interval_end)
                %endif
                {
                    partial_scan_buffer[unit_base + offset] =
                        ldata[offset % K][offset / K];
                }
            }

            // }}}

            local_barrier();
        }

    % endfor

    // write interval sum
    if (LID_0 == 0)
    {
        %if is_first_level:
        interval_results[GID_0] = partial_scan_buffer[interval_end - 1];
        %endif
        %if is_segmented and is_first_level:
            g_first_segment_start_in_interval[GID_0] = first_segment_start_in_interval;
        %endif
    }
}
"""

# }}}

# {{{ update

UPDATE_SOURCE = SHARED_PREAMBLE + r"""//CL//

#define OUTPUT_STMT(i, prev_item, item) { ${output_statement}; }

KERNEL
REQD_WG_SIZE(WG_SIZE, 1, 1)
void ${name_prefix}_final_update(
    ${argument_signature},
    const index_type N,
    const index_type interval_size,
    GLOBAL_MEM scan_type *interval_results,
    GLOBAL_MEM scan_type *partial_scan_buffer
    %if is_segmented:
        , GLOBAL_MEM index_type *g_first_segment_start_in_interval
    %endif
    %if is_segmented and use_lookbehind_update:
        , GLOBAL_MEM char *g_segment_start_flags
    %endif
    )
{
    %if use_lookbehind_update:
        LOCAL_MEM scan_type ldata[WG_SIZE];
    %endif
    %if is_segmented and use_lookbehind_update:
        LOCAL_MEM char l_segment_start_flags[WG_SIZE];
    %endif

    const index_type interval_begin = interval_size * GID_0;
    const index_type interval_end = min(interval_begin + interval_size, N);

    // carry from last interval
    scan_type carry = ${neutral};
    if (GID_0 != 0)
        carry = interval_results[GID_0 - 1];

    %if is_segmented:
        const index_type first_seg_start_in_interval =
            g_first_segment_start_in_interval[GID_0];

    %endif

    %if not use_lookbehind_update:
        // {{{ no look-behind ('prev_item' not in output_statement -> simpler)

        index_type update_i = interval_begin+LID_0;

        <%def name="update_loop_plain(end)">
            for(; update_i < ${end}; update_i += WG_SIZE)
            {
                scan_type partial_val = partial_scan_buffer[update_i];
                scan_type value = SCAN_EXPR(carry, partial_val);

                OUTPUT_STMT(update_i, prev_item_unavailable_with_local_update, value);
            }
        </%def>

        // {{{ update to the first intra-interval segment boundary

        %if is_segmented:
            index_type seg_end = min(first_seg_start_in_interval, interval_end);
            ${update_loop_plain("seg_end")}
            carry = ${neutral};
        %endif

        // }}}

        ${update_loop_plain("interval_end")}

        // }}}
    %else:
        // {{{ allow look-behind ('prev_item' in output_statement -> complicated)

        // We are not allowed to branch across barriers at a granularity smaller
        // than the whole workgroup. Therefore, the for loop is group-global,
        // and there are lots of local ifs.

        index_type group_base = interval_begin;
        scan_type prev_value = carry; // (A)

        for(; group_base < interval_end; group_base += WG_SIZE)
        {
            index_type update_i = group_base+LID_0;

            // load a work group's worth of data
            if (update_i < interval_end)
            {
                scan_type tmp = partial_scan_buffer[update_i];

                %if is_segmented:
                if (update_i < first_seg_start_in_interval)
                %endif
                { tmp = SCAN_EXPR(carry, tmp); }

                ldata[LID_0] = tmp;

                %if is_segmented:
                    l_segment_start_flags[LID_0] = g_segment_start_flags[update_i];
                %endif
            }

            local_barrier();

            // find prev_value
            if (LID_0 != 0)
                prev_value = ldata[LID_0 - 1];
            /*
            else
                prev_value = carry (see (A)) OR last tail (see (B));
            */

            if (update_i < interval_end)
            {
                %if is_segmented:
                    if (l_segment_start_flags[LID_0])
                        prev_value = ${neutral};
                %endif

                scan_type value = ldata[LID_0];
                OUTPUT_STMT(update_i, prev_value, value)
            }

            if (LID_0 == 0)
                prev_value = ldata[WG_SIZE - 1]; // (B)

            local_barrier();
        }

        // }}}
    %endif
}
"""

# }}}

# {{{ driver

# {{{ helpers

def _process_code_for_macro(code):
    if code.startswith("//CL//"):
        code = code[6:]

    if "//" in code:
        raise RuntimeError("end-of-line comments ('//') may not be used in "
                "scan code snippets")

    return code.replace("\n", " \\\n")

def _round_down_to_power_of_2(val):
    result = 2**bitlog2(val)
    if result > val:
        result >>=1

    assert result <= val
    return result

def _parse_args(arguments):
    from pyopencl.tools import parse_c_arg
    return [parse_c_arg(arg) for arg in arguments.split(",")]

def _get_scalar_arg_dtypes(arg_types):
    result = []

    from pyopencl.tools import ScalarArg
    for arg_type in arg_types:
        if isinstance(arg_type, ScalarArg):
            result.append(arg_type.dtype)
        else:
            result.append(None)

    return result

_PREFIX_WORDS = set("""
        ldata partial_scan_buffer global scan_offset
        segment_start_in_k_group carry
        g_first_segment_start_in_interval IS_SEG_START tmp Z
        val l_first_segment_start_in_subtree unit_size
        index_type interval_begin interval_size offset_end K
        SCAN_EXPR do_update WG_SIZE
        first_segment_start_in_k_group scan_type
        segment_start_in_subtree offset interval_results interval_end
        first_segment_start_in_subtree unit_base
        first_segment_start_in_interval k INPUT_EXPR
        prev_group_sum prev pv value partial_val pgs OUTPUT_STMT
        is_seg_start update_i scan_item_at_i seq_i read_i
        l_ o_mod_k o_div_k l_segment_start_flags scan_value sum
        first_seg_start_in_interval g_segment_start_flags
        group_base seg_end my_val

        LID_2 LID_1 LID_0
        LDIM_0 LDIM_1 LDIM_2
        GDIM_0 GDIM_1 GDIM_2
        GID_0 GID_1 GID_2
        """.split())

_IGNORED_WORDS = set("""
        typedef for endfor if void while endwhile endfor endif else const printf
        None return bool n char

        set iteritems len setdefault

        GLOBAL_MEM LOCAL_MEM_ARG WITHIN_KERNEL LOCAL_MEM KERNEL REQD_WG_SIZE
        local_barrier
        CLK_LOCAL_MEM_FENCE OPENCL EXTENSION
        pragma __attribute__ __global __kernel __local
        get_local_size get_local_id cl_khr_fp64 reqd_work_group_size
        get_num_groups barrier get_group_id

        _final_update _scan_intervals _debug_scan

        positions all padded integer its previous write based true writes 0
        has local worth scan_expr to read cannot not X items False bank
        four beginning follows applicable item min each indices works side
        scanning right summed relative used id out index avoid current state
        boundary True across be This reads groups along Otherwise undetermined
        store of times prior s update first regardless Each number because
        array unit from segment conflicts two parallel 2 empty define direction
        CL padding work tree bounds values and adds
        scan is allowed thus it an as enable at in occur sequentially end no
        storage data 1 largest may representable uses entry Y meaningful
        computations interval At the left dimension know d
        A load B group perform shift tail see last OR
        this add fetched into are directly need
        gets them stenciled that undefined
        there up any ones or name only relevant populated
        even wide we Prepare int seg Note re below place take variable must
        intra Therefore find code assumption
        branch workgroup complicated granularity phase remainder than simpler
        We smaller look ifs lots self behind allow barriers whole loop

        is_tail is_first_level input_expr argument_signature preamble
        double_support neutral output_statement index_type_max
        k_group_size name_prefix is_segmented index_ctype scan_ctype
        wg_size is_segment_start_expr fetch_expr_offsets
        arg_ctypes ife_offsets input_fetch_exprs def
        ife_offset arg_name local_fetch_expr_args update_body
        update_loop_lookbehind update_loop_plain update_loop
        use_lookbehind_update store_segment_start_flags
        update_loop first_seg

        a b prev_item i prev_item_unavailable_with_local_update prev_value
        N NO_SEG_BOUNDARY
        """.split())

def _make_template(s):
    leftovers = set()

    def replace_id(match):
        # avoid name clashes with user code by adding 'psc_' prefix to
        # identifiers.

        word = match.group(1)
        if word in _IGNORED_WORDS:
            return word
        elif word in _PREFIX_WORDS:
            return "psc_"+word
        else:
            leftovers.add(word)
            return word

    import re
    s = re.sub(r"\b([a-zA-Z0-9_]+)\b", replace_id, s)

    if leftovers:
        from warnings import warn
        warn("leftover words in identifier prefixing: " + " ".join(leftovers))

    return mako.template.Template(s, strict_undefined=True)

from pytools import Record
class _ScanKernelInfo(Record):
    pass

# }}}

class _GenericScanKernelBase(object):
    # {{{ constructor, argument processing

    def __init__(self, ctx, dtype,
            arguments, input_expr, scan_expr, neutral, output_statement,
            is_segment_start_expr=None, input_fetch_exprs=[],
            index_dtype=np.int32,
            name_prefix="scan", options=[], preamble="", devices=None):
        """
        :arg ctx: a :class:`pyopencl.Context` within which the code
            for this scan kernel will be generated.
        :arg dtype: the :class:`numpy.dtype` with which the scan will
            be performed. May be a structured type if that type was registered
            through :func:`pyopencl.tools.register_dtype`.
        :arg arguments: A string of comma-separated C argument declarations.
            If *arguments* is specified, then *input_expr* must also be
            specified. All types used here must be known to PyOpenCL.
            (see :func:`pyopencl.tools.register_dtype`).
        :arg scan_expr: The associative, binary operation carrying out the scan,
            represented as a C string. Its two arguments are available as `a`
            and `b` when it is evaluated. `b` is guaranteed to be the
            'element being updated', and `a` is the increment. Thus,
            if some data is supposed to just propagate along without being
            modified by the scan, it should live in `b`.

            This expression may call functions given in the *preamble*.
        :arg input_expr: A C expression, encoded as a string, resulting
            in the values to which the scan is applied. This may be used
            to apply a mapping to values stored in *arguments* before being
            scanned. The result of this expression must match *dtype*.
            The index intended to be mapped is available as *i* in this
            expression. This expression may also use the variables defined
            by *input_fetch_expr*.

            This expression may also call functions given in the *preamble*.
        :arg output_statement: a C statement that writes
            the output of the scan. It has access to the scan result as *item*,
            the preceding scan result item as *prev_item*, and the current index
            as *i*. *prev_item* in a segmented scan will be the neutral element
            at a segment boundary, not the immediately preceding item.

            Using *prev_item* in output statement has a small run-time cost.
            *prev_item* enables the construction of an exclusive scan.
        :arg is_segment_start_expr: A C expression, encoded as a string,
            resulting in a C `bool` value that determines whether a new
            scan segments starts at index *i*.  If given, makes the scan a
            segmented scan. Has access to the current index `i`, the result
            of *input_expr* as a, and in addition may use *arguments* and
            *input_fetch_expr* variables just like *input_expr*.

            If it returns true, then previous sums will not spill over into the
            item with index *i* or subsequent items.
        :arg input_fetch_exprs: a list of tuples *(NAME, ARG_NAME, OFFSET)*.
            An entry here has the effect of doing the equivalent of the following
            before input_expr::

                ARG_NAME_TYPE NAME = ARG_NAME[i+OFFSET];

            `OFFSET` is allowed to be 0 or -1, and `ARG_NAME_TYPE` is the type
            of `ARG_NAME`.
        :arg preamble: |preamble|

        The first array in the argument list determines the size of the index
        space over which the scan is carried out, and thus the values over
        which the index *i* occurring in a number of code fragments in
        arguments above will vary.

        All code fragments further have access to N, the number of elements
        being processed in the scan.
        """

        self.context = ctx
        dtype = self.dtype = np.dtype(dtype)

        if neutral is None:
            from warnings import warn
            warn("not specifying 'neutral' is deprecated and will lead to "
                    "wrong results if your scan is not in-place or your "
                    "'output_statement' otherwise does something non-trivial",
                    stacklevel=2)

        self.index_dtype = np.dtype(index_dtype)
        if np.iinfo(self.index_dtype).min >= 0:
            raise TypeError("index_dtype must be signed")

        if devices is None:
            devices = ctx.devices
        self.devices = devices
        self.options = options

        self.arguments = arguments
        self.parsed_args = _parse_args(self.arguments)
        from pyopencl.tools import VectorArg
        self.first_array_idx = [
                i for i, arg in enumerate(self.parsed_args)
                if isinstance(arg, VectorArg)][0]

        self.input_expr = input_expr

        self.is_segment_start_expr = is_segment_start_expr
        self.is_segmented = is_segment_start_expr is not None
        if self.is_segmented:
            is_segment_start_expr = _process_code_for_macro(is_segment_start_expr)

        self.output_statement = output_statement

        for name, arg_name, ife_offset in input_fetch_exprs:
            if ife_offset not in [0, -1]:
                raise RuntimeError("input_fetch_expr offsets must either be 0 or -1")
        self.input_fetch_exprs = input_fetch_exprs

        arg_ctypes = {}
        for arg in self.parsed_args:
            arg_ctypes[arg.name] = dtype_to_ctype(arg.dtype)

        self.options = options
        self.name_prefix = name_prefix

        # {{{ set up shared code dict

        from pytools import all
        from pyopencl.characterize import has_double_support

        self.code_variables = dict(
            preamble=preamble,
            name_prefix=name_prefix,
            index_ctype=dtype_to_ctype(self.index_dtype),
            index_type_max=str(np.iinfo(self.index_dtype).max),
            scan_ctype=dtype_to_ctype(dtype),
            is_segmented=self.is_segmented,
            arg_ctypes=arg_ctypes,
            scan_expr=_process_code_for_macro(scan_expr),
            neutral=_process_code_for_macro(neutral),
            double_support=all(
                has_double_support(dev) for dev in devices),
            )

        # }}}

        self.finish_setup()

    # }}}

class GenericScanKernel(_GenericScanKernelBase):
    """Generates and executes code that performs prefix sums ("scans") on
    arbitrary types, with many possible tweaks.
    """

    def finish_setup(self):
        use_lookbehind_update = "prev_item" in self.output_statement
        self.store_segment_start_flags = self.is_segmented and use_lookbehind_update

        # {{{ loop to find usable workgroup size, build first-level scan

        trip_count = 0

        max_scan_wg_size = min(dev.max_work_group_size for dev in self.devices)

        while True:
            candidate_scan_info = self.build_scan_kernel(
                    max_scan_wg_size, self.arguments, self.input_expr,
                    self.is_segment_start_expr,
                    input_fetch_exprs=self.input_fetch_exprs,
                    is_first_level=True,
                    store_segment_start_flags=self.store_segment_start_flags)

            # Will this device actually let us execute this kernel
            # at the desired work group size? Building it is the
            # only way to find out.
            kernel_max_wg_size = min(
                    candidate_scan_info.kernel.get_work_group_info(
                        cl.kernel_work_group_info.WORK_GROUP_SIZE,
                        dev)
                    for dev in self.devices)

            if candidate_scan_info.wg_size <= kernel_max_wg_size:
                break
            else:
                max_scan_wg_size = kernel_max_wg_size

            trip_count += 1
            assert trip_count <= 2

        self.first_level_scan_info = candidate_scan_info
        assert (_round_down_to_power_of_2(candidate_scan_info.wg_size)
                == candidate_scan_info.wg_size)

        # }}}

        # {{{ build second-level scan

        second_level_arguments = self.arguments.split(",") + [
                "__global %s *interval_sums" % dtype_to_ctype(self.dtype)]
        second_level_build_kwargs = {}
        if self.is_segmented:
            second_level_arguments.append(
                    "__global %s *g_first_segment_start_in_interval_input"
                    % dtype_to_ctype(self.index_dtype))

            # is_segment_start_expr answers the question "should previous sums
            # spill over into this item". And since g_first_segment_start_in_interval_input
            # answers the question if a segment boundary was found in an interval of data,
            # then if not, it's ok to spill over.
            second_level_build_kwargs["is_segment_start_expr"] = \
                    "g_first_segment_start_in_interval_input[i] != NO_SEG_BOUNDARY"
        else:
            second_level_build_kwargs["is_segment_start_expr"] = None

        self.second_level_scan_info = self.build_scan_kernel(
                max_scan_wg_size,
                arguments=", ".join(second_level_arguments),
                input_expr="interval_sums[i]",
                input_fetch_exprs=[],
                is_first_level=False,
                store_segment_start_flags=False,
                **second_level_build_kwargs)

        assert min(
                candidate_scan_info.kernel.get_work_group_info(
                    cl.kernel_work_group_info.WORK_GROUP_SIZE,
                    dev)
                for dev in self.devices) >= max_scan_wg_size

        # }}}

        # {{{ build final update kernel

        self.update_wg_size = min(max_scan_wg_size, 256)

        final_update_tpl = _make_template(UPDATE_SOURCE)
        final_update_src = str(final_update_tpl.render(
            wg_size=self.update_wg_size,
            output_statement=_process_code_for_macro(self.output_statement),
            argument_signature=_process_code_for_macro(self.arguments),
            is_segment_start_expr=self.is_segment_start_expr,
            input_expr=_process_code_for_macro(self.input_expr),
            use_lookbehind_update=use_lookbehind_update,
            **self.code_variables))

        final_update_prg = cl.Program(self.context, final_update_src).build(self.options)
        self.final_update_knl = getattr(
                final_update_prg,
                self.name_prefix+"_final_update")
        update_scalar_arg_dtypes = (
                _get_scalar_arg_dtypes(self.parsed_args)
                + [self.index_dtype, self.index_dtype, None, None])
        if self.is_segmented:
            update_scalar_arg_dtypes.append(None) # g_first_segment_start_in_interval
        if self.store_segment_start_flags:
            update_scalar_arg_dtypes.append(None) # g_segment_start_flags
        self.final_update_knl.set_scalar_arg_dtypes(update_scalar_arg_dtypes)

        # }}}

    # {{{ scan kernel build

    def build_scan_kernel(self, max_wg_size, arguments, input_expr,
            is_segment_start_expr, input_fetch_exprs, is_first_level,
            store_segment_start_flags):
        scalar_arg_dtypes = _get_scalar_arg_dtypes(_parse_args(arguments))

        # Thrust says that 128 is big enough for GT200
        wg_size = _round_down_to_power_of_2(
                min(max_wg_size, 128))

        # k_group_size should be a power of two because of in-kernel
        # division by that number.

        if wg_size < 16:
            # Hello, Apple CPU. Nice to see you.
            k_group_size = 128 # FIXME: guesswork
        else:
            k_group_size = 8

        scan_tpl = _make_template(SCAN_INTERVALS_SOURCE)
        scan_src = str(scan_tpl.render(
            wg_size=wg_size,
            input_expr=input_expr,
            k_group_size=k_group_size,
            argument_signature=_process_code_for_macro(arguments),
            is_segment_start_expr=is_segment_start_expr,
            input_fetch_exprs=input_fetch_exprs,
            is_first_level=is_first_level,
            store_segment_start_flags=store_segment_start_flags,
            **self.code_variables))

        prg = cl.Program(self.context, scan_src).build(self.options)

        knl = getattr(
                prg,
                self.code_variables["name_prefix"]+"_scan_intervals")

        scalar_arg_dtypes.extend(
                (None, self.index_dtype, self. index_dtype))
        if is_first_level:
            scalar_arg_dtypes.append(None) # interval_results
        if self.is_segmented and is_first_level:
            scalar_arg_dtypes.append(None) # g_first_segment_start_in_interval
        if store_segment_start_flags:
            scalar_arg_dtypes.append(None) # g_segment_start_flags
        knl.set_scalar_arg_dtypes(scalar_arg_dtypes)

        return _ScanKernelInfo(
                kernel=knl, wg_size=wg_size, knl=knl, k_group_size=k_group_size)

    # }}}

    def __call__(self, *args, **kwargs):
        # {{{ argument processing

        allocator = kwargs.get("allocator")
        queue = kwargs.get("queue")

        if len(args) != len(self.parsed_args):
            raise TypeError("invalid number of arguments in "
                    "custom-arguments mode")

        first_array = args[self.first_array_idx]
        allocator = allocator or first_array.allocator
        queue = queue or first_array.queue

        n, = first_array.shape

        data_args = []
        from pyopencl.tools import VectorArg
        for arg_descr, arg_val in zip(self.parsed_args, args):
            if isinstance(arg_descr, VectorArg):
                data_args.append(arg_val.data)
            else:
                data_args.append(arg_val)

        # }}}

        l1_info = self.first_level_scan_info
        l2_info = self.second_level_scan_info

        # see CL source above for terminology
        unit_size  = l1_info.wg_size * l1_info.k_group_size
        max_intervals = 3*max(dev.max_compute_units for dev in self.devices)

        from pytools import uniform_interval_splitting
        interval_size, num_intervals = uniform_interval_splitting(
                n, unit_size, max_intervals)

        # {{{ allocate some buffers

        interval_results = cl_array.empty(queue,
                num_intervals, dtype=self.dtype,
                allocator=allocator)

        partial_scan_buffer = cl_array.empty(
                queue, n, dtype=self.dtype,
                allocator=allocator)

        if self.store_segment_start_flags:
            segment_start_flags = cl_array.empty(
                    queue, n, dtype=np.bool,
                    allocator=allocator)

        # }}}

        # {{{ first level scan of interval (one interval per block)

        scan1_args = data_args + [
                partial_scan_buffer.data, n, interval_size, interval_results.data,
                ]

        if self.is_segmented:
            first_segment_start_in_interval = cl_array.empty(queue,
                    num_intervals, dtype=self.index_dtype,
                    allocator=allocator)
            scan1_args.append(first_segment_start_in_interval.data)

        if self.store_segment_start_flags:
            scan1_args.append(segment_start_flags.data)

        l1_info.kernel(
                queue, (num_intervals,), (l1_info.wg_size,),
                *scan1_args, **dict(g_times_l=True))

        # }}}

        # {{{ second level scan of per-interval results

        # can scan at most one interval
        assert interval_size >= num_intervals

        scan2_args = data_args + [
                interval_results.data, # interval_sums
                ]
        if self.is_segmented:
            scan2_args.append(first_segment_start_in_interval.data)
        scan2_args = scan2_args + [
                interval_results.data, # partial_scan_buffer
                num_intervals, interval_size]

        l2_info.kernel(
                queue, (1,), (l1_info.wg_size,),
                *scan2_args, **dict(g_times_l=True))

        # }}}

        # {{{ update intervals with result of interval scan

        upd_args = data_args + [
                n, interval_size, interval_results.data, partial_scan_buffer.data]
        if self.is_segmented:
            upd_args.append(first_segment_start_in_interval.data)
        if self.store_segment_start_flags:
            upd_args.append(segment_start_flags.data)

        self.final_update_knl(
                queue, (num_intervals,), (self.update_wg_size,),
                *upd_args, **dict(g_times_l=True))

        # }}}

# }}}

# {{{ debug kernel

DEBUG_SCAN_TEMPLATE = SHARED_PREAMBLE + """//CL//

#define OUTPUT_STMT(i, prev_item, item) { ${output_statement}; }

KERNEL
REQD_WG_SIZE(1, 1, 1)
void ${name_prefix}_debug_scan(
    ${argument_signature},
    const index_type N)
{
    scan_type item = ${neutral};
    scan_type prev_item;

    for (index_type i = 0; i < N; ++i)
    {
        %for name, arg_name, ife_offset in input_fetch_exprs:
            ${arg_ctypes[arg_name]} ${name};
            %if ife_offset < 0:
                if (i+${ife_offset} >= 0)
                    ${name} = ${arg_name}[i+offset];
            %else:
                ${name} = ${arg_name}[i];
            %endif
        %endfor

        scan_type my_val = INPUT_EXPR(i);

        prev_item = item;
        %if is_segmented:
            {
                bool is_seg_start = IS_SEG_START(i, my_val);
                if (is_seg_start)
                    prev_item = ${neutral};
            }
        %endif
        item = SCAN_EXPR(prev_item, my_val);

        {
            OUTPUT_STMT(i, prev_item, item);
        }
    }
}
"""

class GenericDebugScanKernel(_GenericScanKernelBase):
    def finish_setup(self):
        scan_tpl = _make_template(DEBUG_SCAN_TEMPLATE)
        scan_src = str(scan_tpl.render(
            output_statement=_process_code_for_macro(self.output_statement),
            argument_signature=_process_code_for_macro(self.arguments),
            is_segment_start_expr=self.is_segment_start_expr,
            input_expr=_process_code_for_macro(self.input_expr),
            input_fetch_exprs=self.input_fetch_exprs,
            wg_size=1,
            **self.code_variables))

        scan_prg = cl.Program(self.context, scan_src).build(self.options)
        self.kernel = getattr(
                scan_prg, self.name_prefix+"_debug_scan")
        scalar_arg_dtypes = (
                _get_scalar_arg_dtypes(self.parsed_args)
                + [self.index_dtype])
        self.kernel.set_scalar_arg_dtypes(scalar_arg_dtypes)

        # }}}

    def __call__(self, *args, **kwargs):
        # {{{ argument processing

        allocator = kwargs.get("allocator")
        queue = kwargs.get("queue")

        if len(args) != len(self.parsed_args):
            raise TypeError("invalid number of arguments in "
                    "custom-arguments mode")

        first_array = args[self.first_array_idx]
        allocator = allocator or first_array.allocator
        queue = queue or first_array.queue

        n, = first_array.shape

        data_args = []
        from pyopencl.tools import VectorArg
        for arg_descr, arg_val in zip(self.parsed_args, args):
            if isinstance(arg_descr, VectorArg):
                data_args.append(arg_val.data)
            else:
                data_args.append(arg_val)

        # }}}

        self.kernel(queue, (1,), (1,), *(data_args + [n]))

# }}}

# {{{ compatibility interface

class _LegacyScanKernelBase(GenericScanKernel): # FIXME
    def __init__(self, ctx, dtype,
            scan_expr, neutral=None,
            name_prefix="scan", options=[], preamble="", devices=None):
        scan_ctype = dtype_to_ctype(dtype)
        GenericScanKernel.__init__(self,
                ctx, dtype,
                arguments="__global %s *input_ary, __global %s *output_ary" % (
                    scan_ctype, scan_ctype),
                input_expr="input_ary[i]",
                scan_expr=scan_expr,
                neutral=neutral,
                output_statement=self.ary_output_statement,
                options=options, preamble=preamble, devices=devices)

    def __call__(self, input_ary, output_ary=None, allocator=None, queue=None):
        allocator = allocator or input_ary.allocator
        queue = queue or input_ary.queue or output_ary.queue

        if output_ary is None:
            output_ary = input_ary

        if isinstance(output_ary, (str, unicode)) and output_ary == "new":
            output_ary = cl_array.empty_like(input_ary, allocator=allocator)

        if input_ary.shape != output_ary.shape:
            raise ValueError("input and output must have the same shape")

        if not input_ary.flags.forc:
            raise RuntimeError("ScanKernel cannot "
                    "deal with non-contiguous arrays")

        n, = input_ary.shape

        if not n:
            return output_ary

        GenericScanKernel.__call__(self,
                input_ary, output_ary, allocator=allocator, queue=queue)

        return output_ary

class InclusiveScanKernel(_LegacyScanKernelBase):
    ary_output_statement = "output_ary[i] = item;"

class ExclusiveScanKernel(_LegacyScanKernelBase):
    ary_output_statement = "output_ary[i] = prev_item;"

# }}}

# {{{ higher-level trickery

@context_dependent_memoize
def _get_copy_if_kernel(ctx, dtype, predicate, scan_dtype,
        extra_args_types, preamble):
    ctype = dtype_to_ctype(dtype)
    arguments = [
        "__global %s *ary" % ctype,
        "__global %s *out" % ctype,
        "__global unsigned long *count",
        ] + [
                "%s %s" % (dtype_to_ctype(arg_dtype), name)
                for name, arg_dtype in extra_args_types]

    return GenericScanKernel(
            ctx, dtype,
            arguments=", ".join(arguments),
            input_expr="(%s) ? 1 : 0" % predicate,
            scan_expr="a+b", neutral="0",
            output_statement="""
                if (prev_item != item) out[item-1] = ary[i];
                if (i+1 == N) *count = item;
                """,
            preamble=preamble)

def copy_if(ary, predicate, extra_args=[], queue=None, preamble=""):
    """Copy the elements of *ary* satisfying *predicate* to an output array.

    :arg predicate: a C expression evaluating to a `bool`, represented as a string.
        The value to test is available as `ary[i]`, and if the expression evaluates
        to `true`, then this value ends up in the output.
    :arg extra_args: |scan_extra_args|
    :arg preamble: |preamble|
    :returns: a tuple *(out, count)* where *out* is the output array and *count*
        is an on-device scalar (fetch to host with `count.get()`) indicating
        how many elements satisfied *predicate*.
    """
    if len(ary) > np.iinfo(np.uint32).max:
        scan_dtype = np.uint64
    else:
        scan_dtype = np.uint32

    extra_args_types = tuple((name, val.dtype) for name, val in extra_args)
    extra_args_values = tuple(val for name, val in extra_args)

    knl = _get_copy_if_kernel(ary.context, ary.dtype, predicate, scan_dtype,
            extra_args_types, preamble=preamble)
    out = cl_array.empty_like(ary)
    count = ary._new_with_changes(data=None, shape=(), strides=(), dtype=np.uint64)
    knl(ary, out, count, *extra_args_values, queue=queue)
    return out, count

def remove_if(ary, predicate, extra_args=[], queue=None, preamble=""):
    """Copy the elements of *ary* not satisfying *predicate* to an output array.

    :arg predicate: a C expression evaluating to a `bool`, represented as a string.
        The value to test is available as `ary[i]`, and if the expression evaluates
        to `false`, then this value ends up in the output.
    :arg extra_args: |scan_extra_args|
    :arg preamble: |preamble|
    :returns: a tuple *(out, count)* where *out* is the output array and *count*
        is an on-device scalar (fetch to host with `count.get()`) indicating
        how many elements did not satisfy *predicate*.
    """
    return copy_if(ary, "!(%s)" % predicate, extra_args=extra_args, queue=queue,
            preamble=preamble)

@context_dependent_memoize
def _get_partition_kernel(ctx, dtype, predicate, scan_dtype,
        extra_args_types, preamble):
    ctype = dtype_to_ctype(dtype)
    arguments = [
        "__global %s *ary" % ctype,
        "__global %s *out_true" % ctype,
        "__global %s *out_false" % ctype,
        "__global unsigned long *count_true",
        ] + [
                "%s %s" % (dtype_to_ctype(arg_dtype), name)
                for name, arg_dtype in extra_args_types]

    return GenericScanKernel(
            ctx, dtype,
            arguments=", ".join(arguments),
            input_expr="(%s) ? 1 : 0" % predicate,
            scan_expr="a+b", neutral="0",
            output_statement="""
                if (prev_item != item)
                    out_true[item-1] = ary[i];
                else
                    out_false[i-item] = ary[i];
                if (i+1 == N) *count_true = item;
                """,
            preamble=preamble)

def partition(ary, predicate, extra_args=[], queue=None, preamble=""):
    """Copy the elements of *ary* into one of two arrays depending on whether
    they satisfy *predicate*.

    :arg predicate: a C expression evaluating to a `bool`, represented as a string.
        The value to test is available as `ary[i]`.
    :arg extra_args: |scan_extra_args|
    :arg preamble: |preamble|
    :returns: a tuple *(out_true, out_false, count)* where *count*
        is an on-device scalar (fetch to host with `count.get()`) indicating
        how many elements satisfied the predicate.
    """
    if len(ary) > np.iinfo(np.uint32).max:
        scan_dtype = np.uint64
    else:
        scan_dtype = np.uint32

    extra_args_types = tuple((name, val.dtype) for name, val in extra_args)
    extra_args_values = tuple(val for name, val in extra_args)

    knl = _get_partition_kernel(ary.context, ary.dtype, predicate, scan_dtype,
            extra_args_types, preamble)
    out_true = cl_array.empty_like(ary)
    out_false = cl_array.empty_like(ary)
    count = ary._new_with_changes(data=None, shape=(), strides=(), dtype=np.uint64)
    knl(ary, out_true, out_false, count, *extra_args_values, queue=queue)
    return out_true, out_false, count

@context_dependent_memoize
def _get_unique_kernel(ctx, dtype, is_equal_expr, scan_dtype,
        extra_args_types, preamble):
    ctype = dtype_to_ctype(dtype)
    arguments = [
        "__global %s *ary" % ctype,
        "__global %s *out" % ctype,
        "__global unsigned long *count_unique",
        ] + [
                "%s %s" % (dtype_to_ctype(arg_dtype), name)
                for name, arg_dtype in extra_args_types]

    key_expr_define = "#define IS_EQUAL_EXPR(a, b) %s\n" \
            % _process_code_for_macro(is_equal_expr)
    return GenericScanKernel(
            ctx, dtype,
            arguments=", ".join(arguments),
            input_fetch_exprs=[
                ("ary_im1", "ary", -1),
                ("ary_i", "ary", 0),
                ],
            input_expr="(i == 0) || (IS_EQUAL_EXPR(ary_im1, ary_i) ? 0 : 1)",
            scan_expr="a+b", neutral="0",
            output_statement="""
                if (prev_item != item) out[item-1] = ary[i];
                if (i+1 == N) *count_unique = item;
                """,
            preamble=preamble+"\n\n"+key_expr_define)

def unique(ary, is_equal_expr="a == b", extra_args=[], queue=None, preamble=""):
    """Copy the elements of *ary* into the output if *is_equal_expr*, applied to the
    array element and its predecessor, yields false.

    Works like the UNIX command :program:`uniq`, with a potentially custom comparison.
    This operation is often used on sorted sequences.

    :arg is_equal_expr: a C expression evaluating to a `bool`, represented as a string.
        The elements being compared are available as `a` and `b`. If this expression
        yields `false`, the two are considered distinct.
    :arg extra_args: |scan_extra_args|
    :arg preamble: |preamble|
    :returns: a tuple *(out, count)* where *out* is the output array and *count*
        is an on-device scalar (fetch to host with `count.get()`) indicating
        how many elements satisfied the predicate.
    """

    if len(ary) > np.iinfo(np.uint32).max:
        scan_dtype = np.uint64
    else:
        scan_dtype = np.uint32

    extra_args_types = tuple((name, val.dtype) for name, val in extra_args)
    extra_args_values = tuple(val for name, val in extra_args)

    knl = _get_unique_kernel(ary.context, ary.dtype, is_equal_expr, scan_dtype,
            extra_args_types, preamble)
    out = cl_array.empty_like(ary)
    count = ary._new_with_changes(data=None, shape=(), strides=(), dtype=np.uint64)
    knl(ary, out, count, *extra_args_values, queue=queue)
    return out, count

# }}}

# vim: filetype=pyopencl:fdm=marker
