from __future__ import division, absolute_import

__copyright__ = "Copyright (C) 2009-2013 Andreas Kloeckner"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""


from pymbolic.primitives import Variable, Subscript

reserved = {
        'not': 'NOT',
        'and': 'AND',
        'or': 'OR',
        'if': 'IF',
        'else': 'ELSE',
        'j': 'IOTA'}

tokens = (
        'EQUAL', 'NOTEQUAL', 'LEFTSHIFT', 'RIGHTSHIFT', 'LESSEQUAL',
        'GREATEREQUAL', 'LESS', 'GREATER', 'ASSIGN', 'INT', 'FLOAT1', 'FLOAT2',
        'FLOAT3', 'FLOAT4', 'FLOAT5', 'PLUS', 'MINUS', 'TIMES', 'OVER',
        'POWER', 'FLOORDIV', 'MODULO', 'BITWISEAND', 'BITWISEOR', 'BITWISENOT',
        'BITWISENOT', 'BITWISEXOR', 'OPENPAR', 'CLOSEPAR', 'OPENBRACKET',
        'CLOSEBRACKET', 'COMMA', 'DOT', 'COLON',) + reserved.values()

t_EQUAL = r'=='
t_NOTEQUAL = r'!='

t_LEFTSHIFT = r'<<'
t_LEFTSHIFT = r'>>'

t_LESSEQUAL = r'<='
t_GREATEREQUAL = r'>='

t_LESS = r'<'
t_GREATER = r'>'
t_ASSIGN = r'='

t_INT = r'[0-9]+'

# {{{ floats

# has digits before the dot (after optional)
t_FLOAT1 = r'[0-9]+\.[0-9]*([eEdD][+-]?[0-9]+)?([a-zA-Z]*)'
# has digits after the dot (before optional)
t_FLOAT2 = r'[0-9]+(\.[0-9]*)?[eEdD][+-]?[0-9]+([a-zA-Z]*)'
t_FLOAT3 = r'[0-9]*\.[0-9]+([eEdD][+-]?[0-9]+)?([a-zA-Z]*)'
t_FLOAT4 = r'[0-9]*\.[0-9]+[eEdD][+-]?[0-9]+([a-zA-Z]*)'
# has a letter tag
t_FLOAT5 = r'[0-9]+([a-zA-Z]+)'

# }}}

t_PLUS = r'+'
t_MINUS = r'-'
t_TIMES = r'\*'
t_OVER = r'/'
t_POWER = r'\*\*'
t_FLOORDIV = r'//'
t_MODULO = r'%'

t_BITWISEAND = r'&'
t_BITWISEOR = r'\|'
t_BITWISENOT = r'\~'
t_BITWISEXOR = r'\^'

t_OPENPAR = r'\('
t_CLOSEPAR = r'\)'
t_OPENBRACKET = r'\['
t_CLOSEBRACKET = r'\]'

t_COMMA = r','
t_DOT = r'\.'
t_COLON = r'\.'

t_ignore = ' \t'


def t_IDENTIFIER(t):
    r'[@$a-z_A-Z_][@$a-zA-Z_0-9]*'
    t.type = reserved.get(t.value, 'IDENTIFIER')
    return t


def t_error(t):
    raise RuntimeError("Illegal character '%s'." % t.value[0])


precedence = (
        ('left', 'IF', 'ELSE'),
        ('left', 'OR'),
        ('left', 'AND'),
        ('left', 'NOT'),
        ('nonassoc', 'GREATEREQUAL', 'LESSEQUAL', 'LESS', 'GREATER', 'NOTEQUAL',
            'EQUAL'),
        ('left', 'BITWISEOR'),
        ('left', 'BITWISEXOR'),
        ('left', 'BITWISEAND'),
        ('left', 'LEFTSHIFT', 'RIGHTSHIFT'),
        ('left', 'PLUS', 'MINUS'),
        ('left', 'TIMES', 'OVER', 'FLOORDIV', 'MODULO'),
        ('right', 'UMINUS', 'UPLUS', 'BITWISENOT'),
        ('right', 'POWER'),
        ('right', 'OPENPAR', 'OPENBRACKET'),
        ('nonassoc', 'COLON'),
        ('left', 'DOT', 'CLOSEPAR', 'CLOSEBRACKET'),
        )


# {{{ scalars

def p_scalar_as_real_ints(p):
    'scalar : INT'
    p[0] = int(p[1])


def p_scalar_as_imag_ints(p):
    'scalar : INT IOTA'
    p[0] = int(p[1]) * 1j
    # FIXME: Allows illegal syntax like "1 + 2<space>j"


def p_scalar_as_real_floats(p):
    '''scalar : FLOAT1
              | FLOAT2
              | FLOAT3
              | FLOAT4
              | FLOAT5'''
    p[0] = float(p[1])


def p_scalar_as_imag_floats(p):
    '''scalar : FLOAT1 IOTA
              | FLOAT2 IOTA
              | FLOAT3 IOTA
              | FLOAT4 IOTA
              | FLOAT5 IOTA'''
    p[0] = float(p[1]) * 1j
    # FIXME: Allows illegal syntax like "1 + 2<space>j"

# }}}


def p_expr_categories(p):
    '''expression : variable
                  | subscript
                  | call
                  | slice
                  | tuple
                  | list
                  | scalar'''
    p[0] = p[1]


def p_variable(p):
    'variable : IDENTIFIER'
    p[0] = Variable(p[1])


def p_subscript(p):
    '''subscript : expression OPENBRACKET ge_2_elems CLOSEBRACKET
                 | expression OPENBRACKET expression CLOSEBRACKET'''
    p[0] = p[1][p[3]]
    # FIXME: This allows for tuples/lists as indices.(Even current pymbolic
    # master allows for this). Should this be legal?


def p_tuple_with_ge_2_elems(p):
    '''tuple : OPENPAR ge_2_elems  CLOSEPAR
             | OPENPAR  ge_2_elems COMMA CLOSEPAR'''
    p[0] = p[2]


def p_tuple_with_1_elem(p):
    '''tuple : OPENPAR expression COMMA  CLOSEPAR'''
    p[0] = (p[2],)


def p_commaed_collection_of_exprs_recursive_defn(p):
    '''ge_2_elems : ge_2_elems COMMA expression
       ge_1_elems : ge_1_elems COMMA expression
       kwargs : kwargs COMMA kwarg'''
    p[0] = p[1] + (p[3],)


def p_commaed_collection_of_exprs_double_elem(p):
    '''ge_2_elems : expression COMMA expression'''
    p[0] = (p[1], p[3])


def p_commaed_collection_of_exprs_single_elem(p):
    '''ge_1_elems : expression
       kwargs : kwarg'''
    p[0] = (p[1],)


def p_args(p):
    'args : ge_1_elems'
    p[0] = p[1]


def p_call_only_args(p):
    '''call : expression OPENPAR args CLOSEPAR
            | expression OPENPAR args COMMA CLOSEPAR'''
    p[0] = p[1](*p[3])


def p_call_only_kwargs(p):
    '''call : expression OPENPAR kwargs CLOSEPAR
            | expression OPENPAR kwargs COMMA CLOSEPAR'''
    p[0] = p[1](**dict(p[3]))


def p_call_both_args_kwargs(p):
    '''call : expression OPENPAR args COMMA kwargs CLOSEPAR
            | expression OPENPAR args COMMA kwargs COMMA CLOSEPAR'''
    p[0] = p[1](*p[3], **dict(p[5]))


def p_kwarg(p):
    'kwarg : IDENTIFIER ASSIGN expression'
    p[0] = (p[1], p[3])


def p_if_else(p):
    'expression : expression IF expression ELSE expression'
    raise NotImplementedError()


def p_binary_ops(p):
    '''expression : expression OR expression
                  | expression AND expression
                  | expression NOT expression
                  | expression COMPARISON expression
                  | expression BITWISEOR expression
                  | expression BITWISEXOR expression
                  | expression BITWISEAND expression
                  | expression LEFTSHIFT expression
                  | expression RIGHTSHIFT expression
                  | expression PLUS expression
                  | expression MINUS expression
                  | expression TIMES expression
                  | expression OVER expression
                  | expression FLOORDIV expression
                  | expression MODULO expression
                  | expression POWER expression
                  | expression DOT IDENTIFIER'''
    # is there a sensible way of doing this?
    raise NotImplementedError()


def p_comparison(p):
    '''COMPARISON : GREATEREQUAL
                  | LESSEQUAL
                  | LESS
                  | GREATER
                  | EQUAL
                  | NOTEQUAL'''
    raise NotImplementedError()


def p_slice(p):
    'slice : COLON'
    # THere would be 8 variants of this..
    # we have 3 variants of slice
    raise NotImplementedError()


def p_empty(p):
    'empty :'
    pass
