from __future__ import annotations


__copyright__ = """Copyright (C) 2022 University of Illinois Board of Trustees"""

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

from typing import TYPE_CHECKING

from typing_extensions import override

from pymbolic.mapper import CachedWalkMapper


if TYPE_CHECKING:
    from pymbolic.typing import Expression


__doc__ = """
.. autoclass:: NodeCountMapper
    :show-inheritance:

.. autofunction:: get_num_nodes
"""


# {{{ NodeCountMapper

class NodeCountMapper(CachedWalkMapper[[]]):
    """
    Counts the number of nodes in an expression tree. Nodes that occur
    repeatedly as well as :class:`~pymbolic.primitives.CommonSubexpression`
    nodes are only counted once.

    .. attribute:: count

       The number of nodes.
    """

    count: int

    def __init__(self) -> None:
        super().__init__()
        self.count = 0

    @override
    def post_visit(self, expr: object) -> None:
        self.count += 1


def get_num_nodes(expr: Expression) -> int:
    """
    :returns: the number of nodes in *expr*. Nodes that occur
        repeatedly as well as :class:`~pymbolic.primitives.CommonSubexpression`
        nodes are only counted once.
    """

    ncm = NodeCountMapper()
    ncm(expr)

    return ncm.count

# }}}
