__copyright__ = "Copyright (C) 2019 Timothy A. Smith"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import numpy as np
import numpy.linalg as la  # noqa: F401
import pyopencl as cl  # noqa: F401
import pyopencl.array  # noqa
import pyopencl.tools  # noqa
import pyopencl.clrandom  # noqa
import loopy as lp  # noqa

import sys
import logging

import pytest

import utilities as u
import weno_reference_implementation as ref


# FIXME: is there a better way to divide responsibilities with these fixture classes?
class FluxDataVector:
    nvars = 5
    ndim = 3
    dirs = {"x": 1, "y": 2, "z": 3}
    halo = 3

    def __init__(self, nx, ny, nz, states_str, direction):
        self.direction = self.dirs[direction]
        self.dir_internal = self.direction-1

        self.nx = nx
        self.ny = ny
        self.nz = nz

        self.nxhalo = self.nx + 2*self.halo
        self.nyhalo = self.ny + 2*self.halo
        self.nzhalo = self.nz + 2*self.halo

        self.flux_dims = (self.nvars, self.nx, self.ny, self.nz)

        self.metrics = np.stack(
                [np.stack(
                    [np.stack(
                        [np.identity(self.ndim) for i in range(self.nxhalo)],
                        axis=-1) for j in range(self.nyhalo)],
                    axis=-1) for k in range(self.nzhalo)],
                axis=-1).copy(order="F")
        self.jacobians = np.ones((self.nxhalo, self.nyhalo, self.nzhalo),
                order="F")

        state_pair = self.swap_array_rows(
                u.transposed_array_from_string(states_str), self.direction)
        # FIXME: Move array_from_string stuff outside WindowResults
        #   -- just pass an array & have external utilities that generate
        #   Riemann, sine wave, etc. initial conditions
        # FIXME: Consider handling row swapping outside as well?
        # FIXME: Do we even need to swap rows?
        self.state_pair = self.swap_array_rows(
                u.transposed_array_from_string(states_str), self.dir_internal)
        # NOTE: dimensions are nvars x nxhalo x nyhalo x nzhalo
        self.states = self.fill_from_pair()

        # NOTE: dimensions are nvars x nxhalo x nyhalo x nzhalo
        # FIXME: these should be generalized fluxes
        # FIXME: make a clear distinction between fluxes in physical and
        # generalized coordinates
        npoints = self.nxhalo*self.nyhalo*self.nzhalo
        flat_states = self.states.reshape((self.nvars, npoints))
        self.fluxes = ref.pointwise_fluxes(
                flat_states)[:,:,self.dir_internal].T.reshape(
                        (self.nvars, self.nxhalo, self.nyhalo, self.nzhalo)
                        ).copy(order="F")

        # FIXME: use reference implementation
        # NOTE: dimensions are nvars x nx x ny x nz
        self.flux_derivatives = np.zeros((self.nvars, self.nx, self.ny,
            self.nz), order="F")

    def swap_array_rows(self, arr, d):
        p = self.permutation(d)
        arr[p, :] = arr[[1, 2, 3], :]
        return arr

    def permutation(self, d):
        return [(d-1+i) % 3 + 1 for i in range(3)]

    def fill_from_pair(self):
        d = self.dir_internal
        nx_arr = np.array([self.nxhalo, self.nyhalo, self.nzhalo])
        result = u.expand_to_n(self.state_pair, nx_arr[d])

        for i in range(d):
            result = self.add_dimension(result, nx_arr[i])
            result = np.swapaxes(result, -2, -1)
        for i in range(d+1,self.ndim):
            result = self.add_dimension(result, nx_arr[i])

        return result.copy(order="F")

    def add_dimension(self, arr, n):
        return np.stack([arr for i in range(n)], axis=-1)


vector_data = {}

vector_data["Case flat:x"] = FluxDataVector(
        nx=6, ny=2, nz=2,
        states_str="1 1 1 1 5.5,1 1 1 1 5.5",
        direction="x")
vector_data["Case flat:y"] = FluxDataVector(
        nx=2, ny=6, nz=2,
        states_str="1 1 1 1 5.5,1 1 1 1 5.5",
        direction="y")
vector_data["Case flat:z"] = FluxDataVector(
        nx=2, ny=2, nz=6,
        states_str="1 1 1 1 5.5,1 1 1 1 5.5",
        direction="z")

vector_data["Case a:x"] = FluxDataVector(
        nx=6, ny=2, nz=2,
        states_str="2 4 4 4 20,1 1 1 1 5.5",
        direction="x")
vector_data["Case a:y"] = FluxDataVector(
        nx=2, ny=6, nz=2,
        states_str="2 4 4 4 20,1 1 1 1 5.5",
        direction="y")
vector_data["Case a:z"] = FluxDataVector(
        nx=2, ny=2, nz=6,
        states_str="2 4 4 4 20,1 1 1 1 5.5",
        direction="z")


@pytest.fixture(scope="session", params=[
    "Case flat:x", "Case flat:y", "Case flat:z"])
def cfd_test_data_fixture(request):
    return vector_data[request.param]


def test_compute_flux_derivatives_uniform_grid(queue, cfd_test_data_fixture):
    data = cfd_test_data_fixture

    prg = u.get_weno_program()
    prg = prg.copy(target=lp.PyOpenCLTarget(queue.device))

    flux_derivatives_dev = u.empty_array_on_device(queue, *data.flux_dims)

    prg(queue,
            nvars=data.nvars,
            ndim=data.ndim,
            nx=data.nx,
            ny=data.ny,
            nz=data.nz,
            d=data.direction,
            states=data.states,
            fluxes=data.fluxes,
            metrics=data.metrics,
            metric_jacobians=data.jacobians,
            flux_derivatives=flux_derivatives_dev)

    u.compare_arrays(flux_derivatives_dev.get(), data.flux_derivatives)


# This lets you run 'python test.py test_case(cl._csc)' without pytest.
if __name__ == "__main__":
    if len(sys.argv) > 1:
        logging.basicConfig(level="INFO")
        exec(sys.argv[1])
    else:
        pytest.main([__file__])
