import pytest
import utilities as u


# {{{ FluxDataSingle

class FluxDataSingle:
    nvars = 5
    ndim = 3
    dirs = {"x": 1, "y": 2, "z": 3}

    def __init__(self, states_str, fluxes_str,
            lam_str, R_str, R_inv_str, wavespeeds_str,
            char_fluxes_pos_str, char_fluxes_neg_str,
            weno_weights_pos_str, weno_weights_neg_str,
            consistent_str, dissipation_pos_str, dissipation_neg_str,
            weno_flux_str, direction):

        self.direction = self.dirs[direction]

        self.state_pair = self.swap_array_rows(
                u.transposed_array_from_string(states_str), self.direction)
        self.states = u.expand_to_6(self.state_pair)

        self.flux_pair = self.swap_array_rows(
                u.transposed_array_from_string(fluxes_str), self.direction)
        self.fluxes = u.expand_to_6(self.flux_pair)

        self.lam_pointwise = u.expand_to_6(
                u.transposed_array_from_string(lam_str))

        self.R = self.swap_array_rows(
                u.array_from_string(R_str), self.direction)
        self.R_inv = self.swap_array_cols(
                u.array_from_string(R_inv_str), self.direction)
        self.wavespeeds = u.array_from_string(wavespeeds_str)

        self.char_fluxes_pos = u.expand_to_6(
                u.transposed_array_from_string(char_fluxes_pos_str))
        self.char_fluxes_neg = u.expand_to_6(
                u.transposed_array_from_string(char_fluxes_neg_str))

        self.weno_weights_pos = u.transposed_array_from_string(weno_weights_pos_str)
        self.weno_weights_neg = u.transposed_array_from_string(weno_weights_neg_str)

        self.consistent = self.swap_array(
                u.array_from_string(consistent_str), self.direction)
        self.dissipation_pos = self.swap_array(
                u.array_from_string(dissipation_pos_str), self.direction)
        self.dissipation_neg = self.swap_array(
                u.array_from_string(dissipation_neg_str), self.direction)

        self.weno_flux = self.swap_array(
                u.array_from_string(weno_flux_str), self.direction)

    def swap_array(self, arr, d):
        p = self.permutation(d)
        arr[p] = arr[[1, 2, 3]]
        return arr

    def swap_array_rows(self, arr, d):
        p = self.permutation(d)
        arr[p, :] = arr[[1, 2, 3], :]
        return arr

    def swap_array_cols(self, arr, d):
        p = self.permutation(d)
        arr[:, p] = arr[:, [1, 2, 3]]
        return arr

    def permutation(self, d):
        return [(d-1+i) % 3 + 1 for i in range(3)]

# }}}


single_data = {}

# {{{ Input data: Case (a)

single_data["Case a:x"] = FluxDataSingle(
        states_str="2 4 4 4 20,1 1 1 1 5.5",
        fluxes_str="4 11.2 8 8 46.4,1 2.6 1 1 7.1",
        lam_str=("2. 2. 2. 3.4966629547095764 0.5033370452904236,"
            "1. 1. 1. 2.4966629547095764 -0.49666295470957644"),
        R_str=("1 0 0 0.45781245952886396 0.45781245952886396,"
            "1.5857864376269053 0 0 1.4330995704840366 0.018886008110941373,"
            "1.5857864376269053 1.4142135623730951 0 0.7259927892974889 "
            "0.7259927892974889,"
            "1.5857864376269053 0 1.4142135623730951 0.7259927892974889 "
            "0.7259927892974889,"
            "3.7720779386421466 2.2426406871192857 2.2426406871192857 "
            "5.578600290173388 3.335959603054103"),
        R_inv_str=("0.36752136386566203 0.265894835574803 0.265894835574803 "
            "0.265894835574803 -0.16767379847989416,"
            "-1.1213203435596428 0 0.7071067811865475 0 0,"
            "-1.1213203435596428 0 0 0.7071067811865475 0,"
            "-0.43055863210991147 0.41670966546755195 -0.2903971157189956 "
            "-0.2903971157189956 0.1831249838115456,"
            "1.8120820550093746 -0.9975038969055432 -0.2903971157189956 "
            "-0.2903971157189956 0.1831249838115456"),
        wavespeeds_str="2.2 2.2 2.2 3.8463292501805344 0.553670749819466",
        char_fluxes_pos_str=("1.0907156303492833 1.2301515190164993 "
            "1.2301515190164993 7.523052586013577 0.23295627081394965,"
            "0.4673767959969717 -0.6627416997969526 "
            "-0.6627416997969526 1.4795302623674356 0.3125273038443238"),
        char_fluxes_neg_str=("-0.16835489671908455 -0.05857864376269045 "
            "-0.05857864376269045 -0.7274934638648571 -0.30602629876675014,"
            "-0.06722315769446469 0.24852813742385726 "
            "0.24852813742385726 -0.10725061063772967 -0.37456222716537935"),
        weno_weights_pos_str=("0.999999997585736 0.9999999999716082 "
            "0.9999999999716082 0.9999999999997268 0.9999909282503908,"
            "2.2354258683870077e-9 2.6288602366075745e-11 "
            "2.6288602366075745e-11 2.5299566294114733e-13 8.39888391917283e-6,"
            "1.788382117901561e-10 2.1030934718853924e-12 "
            "2.1030934718853924e-12 2.0239658022589593e-14 6.728656898188601e-7"),
        weno_weights_neg_str=("0.9999965203327451 0.999999959029252 "
            "0.999999959029252 0.9999999975371708 0.9999835300215946,"
            "3.2217041402391125e-6 3.793560951911702e-8 "
            "3.793560951911702e-8 2.2803933579018906e-9 0.000015247816238787948,"
            "2.579631146567429e-7 3.0351383606886035e-9 "
            "3.0351383606886035e-9 1.824357365679662e-10 1.2221621667041045e-6"),
        consistent_str="2.5 6.9 4.5 4.5 26.75",
        dissipation_pos_str=("1.6768551393982627 4.8239743723334865 "
            "3.997611768980551 3.997611768980551 22.145196359962693"),
        dissipation_neg_str=("0.1768550804014161 0.523974175351619 "
            "0.4976116690805634 0.4976116690805634 2.495196349669178"),
        weno_flux_str=("4.353710219799678 12.247948547685105 8.995223438061114 "
            "8.995223438061114 51.39039270963187"),
        direction="x")
single_data["Case a:y"] = FluxDataSingle(
        states_str="2 4 4 4 20,1 1 1 1 5.5",
        fluxes_str="4 11.2 8 8 46.4,1 2.6 1 1 7.1",
        lam_str=("2. 2. 2. 3.4966629547095764 0.5033370452904236,"
            "1. 1. 1. 2.4966629547095764 -0.49666295470957644"),
        R_str=("1 0 0 0.45781245952886396 0.45781245952886396,"
            "1.5857864376269053 0 0 1.4330995704840366 0.018886008110941373,"
            "1.5857864376269053 1.4142135623730951 0 0.7259927892974889 "
            "0.7259927892974889,"
            "1.5857864376269053 0 1.4142135623730951 0.7259927892974889 "
            "0.7259927892974889,"
            "3.7720779386421466 2.2426406871192857 2.2426406871192857 "
            "5.578600290173388 3.335959603054103"),
        R_inv_str=("0.36752136386566203 0.265894835574803 0.265894835574803 "
            "0.265894835574803 -0.16767379847989416,"
            "-1.1213203435596428 0 0.7071067811865475 0 0,"
            "-1.1213203435596428 0 0 0.7071067811865475 0,"
            "-0.43055863210991147 0.41670966546755195 -0.2903971157189956 "
            "-0.2903971157189956 0.1831249838115456,"
            "1.8120820550093746 -0.9975038969055432 -0.2903971157189956 "
            "-0.2903971157189956 0.1831249838115456"),
        wavespeeds_str="2.2 2.2 2.2 3.8463292501805344 0.553670749819466",
        char_fluxes_pos_str=("1.0907156303492833 1.2301515190164993 "
            "1.2301515190164993 7.523052586013577 0.23295627081394965,"
            "0.4673767959969717 -0.6627416997969526 "
            "-0.6627416997969526 1.4795302623674356 0.3125273038443238"),
        char_fluxes_neg_str=("-0.16835489671908455 -0.05857864376269045 "
            "-0.05857864376269045 -0.7274934638648571 -0.30602629876675014,"
            "-0.06722315769446469 0.24852813742385726 "
            "0.24852813742385726 -0.10725061063772967 -0.37456222716537935"),
        weno_weights_pos_str=("0.999999997585736 0.9999999999716082 "
            "0.9999999999716082 0.9999999999997268 0.9999909282503908,"
            "2.2354258683870077e-9 2.6288602366075745e-11 "
            "2.6288602366075745e-11 2.5299566294114733e-13 8.39888391917283e-6,"
            "1.788382117901561e-10 2.1030934718853924e-12 "
            "2.1030934718853924e-12 2.0239658022589593e-14 6.728656898188601e-7"),
        weno_weights_neg_str=("0.9999965203327451 0.999999959029252 "
            "0.999999959029252 0.9999999975371708 0.9999835300215946,"
            "3.2217041402391125e-6 3.793560951911702e-8 "
            "3.793560951911702e-8 2.2803933579018906e-9 0.000015247816238787948,"
            "2.579631146567429e-7 3.0351383606886035e-9 "
            "3.0351383606886035e-9 1.824357365679662e-10 1.2221621667041045e-6"),
        consistent_str="2.5 6.9 4.5 4.5 26.75",
        dissipation_pos_str=("1.6768551393982627 4.8239743723334865 "
            "3.997611768980551 3.997611768980551 22.145196359962693"),
        dissipation_neg_str=("0.1768550804014161 0.523974175351619 "
            "0.4976116690805634 0.4976116690805634 2.495196349669178"),
        weno_flux_str=("4.353710219799678 12.247948547685105 8.995223438061114 "
            "8.995223438061114 51.39039270963187"),
        direction="y")
single_data["Case a:z"] = FluxDataSingle(
        states_str="2 4 4 4 20,1 1 1 1 5.5",
        fluxes_str="4 11.2 8 8 46.4,1 2.6 1 1 7.1",
        lam_str=("2. 2. 2. 3.4966629547095764 0.5033370452904236,"
            "1. 1. 1. 2.4966629547095764 -0.49666295470957644"),
        R_str=("1 0 0 0.45781245952886396 0.45781245952886396,"
            "1.5857864376269053 0 0 1.4330995704840366 0.018886008110941373,"
            "1.5857864376269053 1.4142135623730951 0 0.7259927892974889 "
            "0.7259927892974889,"
            "1.5857864376269053 0 1.4142135623730951 0.7259927892974889 "
            "0.7259927892974889,"
            "3.7720779386421466 2.2426406871192857 2.2426406871192857 "
            "5.578600290173388 3.335959603054103"),
        R_inv_str=("0.36752136386566203 0.265894835574803 0.265894835574803 "
            "0.265894835574803 -0.16767379847989416,"
            "-1.1213203435596428 0 0.7071067811865475 0 0,"
            "-1.1213203435596428 0 0 0.7071067811865475 0,"
            "-0.43055863210991147 0.41670966546755195 -0.2903971157189956 "
            "-0.2903971157189956 0.1831249838115456,"
            "1.8120820550093746 -0.9975038969055432 -0.2903971157189956 "
            "-0.2903971157189956 0.1831249838115456"),
        wavespeeds_str="2.2 2.2 2.2 3.8463292501805344 0.553670749819466",
        char_fluxes_pos_str=("1.0907156303492833 1.2301515190164993 "
            "1.2301515190164993 7.523052586013577 0.23295627081394965,"
            "0.4673767959969717 -0.6627416997969526 "
            "-0.6627416997969526 1.4795302623674356 0.3125273038443238"),
        char_fluxes_neg_str=("-0.16835489671908455 -0.05857864376269045 "
            "-0.05857864376269045 -0.7274934638648571 -0.30602629876675014,"
            "-0.06722315769446469 0.24852813742385726 "
            "0.24852813742385726 -0.10725061063772967 -0.37456222716537935"),
        weno_weights_pos_str=("0.999999997585736 0.9999999999716082 "
            "0.9999999999716082 0.9999999999997268 0.9999909282503908,"
            "2.2354258683870077e-9 2.6288602366075745e-11 "
            "2.6288602366075745e-11 2.5299566294114733e-13 8.39888391917283e-6,"
            "1.788382117901561e-10 2.1030934718853924e-12 "
            "2.1030934718853924e-12 2.0239658022589593e-14 6.728656898188601e-7"),
        weno_weights_neg_str=("0.9999965203327451 0.999999959029252 "
            "0.999999959029252 0.9999999975371708 0.9999835300215946,"
            "3.2217041402391125e-6 3.793560951911702e-8 "
            "3.793560951911702e-8 2.2803933579018906e-9 0.000015247816238787948,"
            "2.579631146567429e-7 3.0351383606886035e-9 "
            "3.0351383606886035e-9 1.824357365679662e-10 1.2221621667041045e-6"),
        consistent_str="2.5 6.9 4.5 4.5 26.75",
        dissipation_pos_str=("1.6768551393982627 4.8239743723334865 "
            "3.997611768980551 3.997611768980551 22.145196359962693"),
        dissipation_neg_str=("0.1768550804014161 0.523974175351619 "
            "0.4976116690805634 0.4976116690805634 2.495196349669178"),
        weno_flux_str=("4.353710219799678 12.247948547685105 8.995223438061114 "
            "8.995223438061114 51.39039270963187"),
        direction="z")

# }}}

# {{{ Input data: Case (b)

single_data["Case b:x"] = FluxDataSingle(
        states_str="1 -1 -1 -1 5.5,2 -4 -4 -4 20",
        fluxes_str="-1 2.6 1 1 -7.1,-4 11.2 8 8 -46.4",
        lam_str=("-1. -1. -1. 0.49666295470957644 -2.4966629547095764,"
            "-2. -2. -2. -0.5033370452904236 -3.4966629547095764"),
        R_str=("1 0 0 0.45781245952886396 0.45781245952886396,"
            "-1.5857864376269053 0 0 -0.018886008110941373 "
            "-1.4330995704840366,"
            "-1.5857864376269053 1.4142135623730951 0 -0.7259927892974889 "
            "-0.7259927892974889,"
            "-1.5857864376269053 0 1.4142135623730951 -0.7259927892974889 "
            "-0.7259927892974889,"
            "3.7720779386421466 -2.2426406871192857 -2.2426406871192857 "
            "3.335959603054103 5.578600290173388"),
        R_inv_str=("0.36752136386566203 -0.265894835574803 -0.265894835574803 "
            "-0.265894835574803 -0.16767379847989416,"
            "1.1213203435596428 0 0.7071067811865475 0 0,"
            "1.1213203435596428 0 0 0.7071067811865475 0,"
            "1.8120820550093746 0.9975038969055432 0.2903971157189956 "
            "0.2903971157189956 0.1831249838115456,"
            "-0.43055863210991147 -0.41670966546755195 0.2903971157189956 "
            "0.2903971157189956 0.1831249838115456"),
        wavespeeds_str="2.2 2.2 2.2 0.553670749819466 3.8463292501805344",
        char_fluxes_pos_str=("0.06722315769446469 0.24852813742385726 "
            "0.24852813742385726 0.37456222716537935 0.10725061063772967,"
            "0.16835489671908455 -0.05857864376269045 "
            "-0.05857864376269045 0.30602629876675014 0.7274934638648571"),
        char_fluxes_neg_str=("-0.4673767959969717 -0.6627416997969526 "
            "-0.6627416997969526 -0.3125273038443238 -1.4795302623674356,"
            "-1.0907156303492833 1.2301515190164993 "
            "1.2301515190164993 -0.23295627081394965 -7.523052586013577"),
        weno_weights_pos_str=("0.9999965203327451 0.999999959029252 "
            "0.999999959029252 0.9999835300215946 0.9999999975371708,"
            "3.2217041402391125e-6 3.793560951911702e-8 "
            "3.793560951911702e-8 0.000015247816238787948 2.2803933579018906e-9,"
            "2.579631146567429e-7 3.0351383606886035e-9 "
            "3.0351383606886035e-9 1.2221621667041045e-6 1.824357365679662e-10"),
        weno_weights_neg_str=("0.999999997585736 0.9999999999716082 "
            "0.9999999999716082 0.9999909282503908 0.9999999999997268,"
            "2.2354258683870077e-9 2.6288602366075745e-11 "
            "2.6288602366075745e-11 8.39888391917283e-6 2.5299566294114733e-13,"
            "1.788382117901561e-10 2.1030934718853924e-12 "
            "2.1030934718853924e-12 6.728656898188601e-7 2.0239658022589593e-14"),
        consistent_str="-2.5 6.899999999999999 4.5 4.5 -26.75",
        dissipation_pos_str=("-0.17685508040141606 0.523974175351619 "
            "0.49761166908056337 0.49761166908056337 -2.495196349669178"),
        dissipation_neg_str=("-1.6768551393982627 4.823974372333487 "
            "3.9976117689805504 3.9976117689805504 -22.145196359962693"),
        weno_flux_str=("-4.353710219799678 12.247948547685105 "
            "8.995223438061114 8.995223438061114 -51.390392709631875"),
        direction="x")
single_data["Case b:y"] = FluxDataSingle(
        states_str="1 -1 -1 -1 5.5,2 -4 -4 -4 20",
        fluxes_str="-1 2.6 1 1 -7.1,-4 11.2 8 8 -46.4",
        lam_str=("-1. -1. -1. 0.49666295470957644 -2.4966629547095764,"
            "-2. -2. -2. -0.5033370452904236 -3.4966629547095764"),
        R_str=("1 0 0 0.45781245952886396 0.45781245952886396,"
            "-1.5857864376269053 0 0 -0.018886008110941373 "
            "-1.4330995704840366,"
            "-1.5857864376269053 1.4142135623730951 0 -0.7259927892974889 "
            "-0.7259927892974889,"
            "-1.5857864376269053 0 1.4142135623730951 -0.7259927892974889 "
            "-0.7259927892974889,"
            "3.7720779386421466 -2.2426406871192857 -2.2426406871192857 "
            "3.335959603054103 5.578600290173388"),
        R_inv_str=("0.36752136386566203 -0.265894835574803 -0.265894835574803 "
            "-0.265894835574803 -0.16767379847989416,"
            "1.1213203435596428 0 0.7071067811865475 0 0,"
            "1.1213203435596428 0 0 0.7071067811865475 0,"
            "1.8120820550093746 0.9975038969055432 0.2903971157189956 "
            "0.2903971157189956 0.1831249838115456,"
            "-0.43055863210991147 -0.41670966546755195 0.2903971157189956 "
            "0.2903971157189956 0.1831249838115456"),
        wavespeeds_str="2.2 2.2 2.2 0.553670749819466 3.8463292501805344",
        char_fluxes_pos_str=("0.06722315769446469 0.24852813742385726 "
            "0.24852813742385726 0.37456222716537935 0.10725061063772967,"
            "0.16835489671908455 -0.05857864376269045 "
            "-0.05857864376269045 0.30602629876675014 0.7274934638648571"),
        char_fluxes_neg_str=("-0.4673767959969717 -0.6627416997969526 "
            "-0.6627416997969526 -0.3125273038443238 -1.4795302623674356,"
            "-1.0907156303492833 1.2301515190164993 "
            "1.2301515190164993 -0.23295627081394965 -7.523052586013577"),
        weno_weights_pos_str=("0.9999965203327451 0.999999959029252 "
            "0.999999959029252 0.9999835300215946 0.9999999975371708,"
            "3.2217041402391125e-6 3.793560951911702e-8 "
            "3.793560951911702e-8 0.000015247816238787948 2.2803933579018906e-9,"
            "2.579631146567429e-7 3.0351383606886035e-9 "
            "3.0351383606886035e-9 1.2221621667041045e-6 1.824357365679662e-10"),
        weno_weights_neg_str=("0.999999997585736 0.9999999999716082 "
            "0.9999999999716082 0.9999909282503908 0.9999999999997268,"
            "2.2354258683870077e-9 2.6288602366075745e-11 "
            "2.6288602366075745e-11 8.39888391917283e-6 2.5299566294114733e-13,"
            "1.788382117901561e-10 2.1030934718853924e-12 "
            "2.1030934718853924e-12 6.728656898188601e-7 2.0239658022589593e-14"),
        consistent_str="-2.5 6.899999999999999 4.5 4.5 -26.75",
        dissipation_pos_str=("-0.17685508040141606 0.523974175351619 "
            "0.49761166908056337 0.49761166908056337 -2.495196349669178"),
        dissipation_neg_str=("-1.6768551393982627 4.823974372333487 "
            "3.9976117689805504 3.9976117689805504 -22.145196359962693"),
        weno_flux_str=("-4.353710219799678 12.247948547685105 "
            "8.995223438061114 8.995223438061114 -51.390392709631875"),
        direction="y")
single_data["Case b:z"] = FluxDataSingle(
        states_str="1 -1 -1 -1 5.5,2 -4 -4 -4 20",
        fluxes_str="-1 2.6 1 1 -7.1,-4 11.2 8 8 -46.4",
        lam_str=("-1. -1. -1. 0.49666295470957644 -2.4966629547095764,"
            "-2. -2. -2. -0.5033370452904236 -3.4966629547095764"),
        R_str=("1 0 0 0.45781245952886396 0.45781245952886396,"
            "-1.5857864376269053 0 0 -0.018886008110941373 "
            "-1.4330995704840366,"
            "-1.5857864376269053 1.4142135623730951 0 -0.7259927892974889 "
            "-0.7259927892974889,"
            "-1.5857864376269053 0 1.4142135623730951 -0.7259927892974889 "
            "-0.7259927892974889,"
            "3.7720779386421466 -2.2426406871192857 -2.2426406871192857 "
            "3.335959603054103 5.578600290173388"),
        R_inv_str=("0.36752136386566203 -0.265894835574803 -0.265894835574803 "
            "-0.265894835574803 -0.16767379847989416,"
            "1.1213203435596428 0 0.7071067811865475 0 0,"
            "1.1213203435596428 0 0 0.7071067811865475 0,"
            "1.8120820550093746 0.9975038969055432 0.2903971157189956 "
            "0.2903971157189956 0.1831249838115456,"
            "-0.43055863210991147 -0.41670966546755195 0.2903971157189956 "
            "0.2903971157189956 0.1831249838115456"),
        wavespeeds_str="2.2 2.2 2.2 0.553670749819466 3.8463292501805344",
        char_fluxes_pos_str=("0.06722315769446469 0.24852813742385726 "
            "0.24852813742385726 0.37456222716537935 0.10725061063772967,"
            "0.16835489671908455 -0.05857864376269045 "
            "-0.05857864376269045 0.30602629876675014 0.7274934638648571"),
        char_fluxes_neg_str=("-0.4673767959969717 -0.6627416997969526 "
            "-0.6627416997969526 -0.3125273038443238 -1.4795302623674356,"
            "-1.0907156303492833 1.2301515190164993 "
            "1.2301515190164993 -0.23295627081394965 -7.523052586013577"),
        weno_weights_pos_str=("0.9999965203327451 0.999999959029252 "
            "0.999999959029252 0.9999835300215946 0.9999999975371708,"
            "3.2217041402391125e-6 3.793560951911702e-8 "
            "3.793560951911702e-8 0.000015247816238787948 2.2803933579018906e-9,"
            "2.579631146567429e-7 3.0351383606886035e-9 "
            "3.0351383606886035e-9 1.2221621667041045e-6 1.824357365679662e-10"),
        weno_weights_neg_str=("0.999999997585736 0.9999999999716082 "
            "0.9999999999716082 0.9999909282503908 0.9999999999997268,"
            "2.2354258683870077e-9 2.6288602366075745e-11 "
            "2.6288602366075745e-11 8.39888391917283e-6 2.5299566294114733e-13,"
            "1.788382117901561e-10 2.1030934718853924e-12 "
            "2.1030934718853924e-12 6.728656898188601e-7 2.0239658022589593e-14"),
        consistent_str="-2.5 6.899999999999999 4.5 4.5 -26.75",
        dissipation_pos_str=("-0.17685508040141606 0.523974175351619 "
            "0.49761166908056337 0.49761166908056337 -2.495196349669178"),
        dissipation_neg_str=("-1.6768551393982627 4.823974372333487 "
            "3.9976117689805504 3.9976117689805504 -22.145196359962693"),
        weno_flux_str=("-4.353710219799678 12.247948547685105 "
            "8.995223438061114 8.995223438061114 -51.390392709631875"),
        direction="z")

# }}}

# {{{ Input data: Case (c)

single_data["Case c:x"] = FluxDataSingle(
        states_str="2 4 8 12 64,1 1 2 3 11",
        fluxes_str="4 11.2 16 24 134.4,1 2.6 2 3 12.6",
        lam_str=("2. 2. 2. 3.4966629547095764 0.5033370452904236,"
            "1. 1. 1. 2.4966629547095764 -0.49666295470957644"),
        R_str=("1 0 0 0.41384589551844686 0.41384589551844686,"
            "1.5857864376269053 0 0 1.363377989567262 -0.05083557280583326,"
            "3.1715728752538106 1.4142135623730951 0 1.3125424167614286 "
            "1.3125424167614286,"
            "4.757359312880716 0 1.4142135623730951 1.968813625142143 "
            "1.968813625142143,"
            "17.60303038033002 4.485281374238571 6.727922061357858 "
            "11.42671019719969 9.184069510080404"),
        R_inv_str=("-1.4118746341171051 0.21727611674823194 0.4345522334964639 "
            "0.6518283502446959 -0.13701474018997217,"
            "-2.2426406871192857 0 0.7071067811865475 0 0,"
            "-3.3639610306789285 0 0 0.7071067811865475 0,"
            "1.7926564050747988 0.4445982978342618 -0.5250169667045714 "
            "-0.7875254500568571 0.16553835820737872,"
            "4.035297092194084 -0.9696152645388333 -0.5250169667045714 "
            "-0.7875254500568571 0.16553835820737872"),
        wavespeeds_str="2.2 2.2 2.2 3.8463292501805344 0.553670749819466",
        char_fluxes_pos_str=("1.1162114029572359 2.4603030380329987 "
            "3.690454557049497 7.986924884679773 0.4290108979594782,"
            "0.26073527447612554 -1.3254833995939053 "
            "-1.9882250993908572 2.052422890589809 0.5017887559695788"),
        char_fluxes_neg_str=("-0.1482823715615963 -0.1171572875253809 "
            "-0.1757359312880714 -0.8893252542028551 -0.20004041758408686,"
            "-0.009488213714461069 0.49705627484771453 "
            "0.7455844122715716 -0.4306378813126712 -0.31431832174320373"),
        weno_weights_pos_str=("0.999999999319454 0.9999999999982254 "
            "0.9999999999996494 0.9999999999997061 0.9999870425235151,"
            "6.301345524776077e-10 1.6430428067143077e-12 "
            "3.245518542322869e-13 2.721049684463335e-13 0.000011996153712066484,"
            "5.041138413805998e-11 1.3144350707803127e-13 "
            "2.5964155584975223e-14 2.1768403038595738e-14 9.613227729623656e-7"),
        weno_weights_neg_str=("0.9999990185022956 0.9999999974390363 "
            "0.9999999994941201 0.9999999917661908 0.9999978651320311,"
            "9.08762722250494e-7 2.3712585027633853e-9 "
            "4.684070887233427e-10 7.62387333908512e-9 1.976628702372801e-6,"
            "7.27349821618269e-8 1.897052057748541e-10 "
            "3.747296441221644e-11 6.099359570650648e-10 1.5823926647977797e-7"),
        consistent_str="2.5 6.9 9. 13.5 73.5",
        dissipation_pos_str=("1.6406634409601506 4.725635754575325 "
            "7.88043892893644 11.820658393408754 68.69422377699841"),
        dissipation_neg_str=("0.14066328824586258 0.4256356884430299 "
            "0.8804384437989111 1.3206576666570364 7.7942206041633595"),
        weno_flux_str=("4.281326729206013 12.051271443018354 "
            "17.76087737273535 26.64131606006579 149.98844438116177"),
        direction="x")
single_data["Case c:y"] = FluxDataSingle(
        states_str="2 8 12 4 64,1 2 3 1 11",
        fluxes_str="8 35.2 48 16 268.8,2 5.6 6 2 25.2",
        lam_str=("4. 4. 4. 5.496662954709576 2.5033370452904236,"
            "2. 2. 2. 3.4966629547095764 0.5033370452904236"),
        R_str=("1 0 0 0.41384589551844686 0.41384589551844686,"
            "3.1715728752538106 0 0 2.019649197947976 0.605435635574881,"
            "4.757359312880716 1.4142135623730951 0 1.968813625142143 "
            "1.968813625142143,"
            "1.5857864376269053 0 1.4142135623730951 0.6562712083807143 "
            "0.6562712083807143,"
            "17.60303038033002 6.727922061357858 2.2426406871192857 "
            "12.548030540759331 8.062749166520762"),
        R_inv_str=("-1.4118746341171051 0.4345522334964639 0.6518283502446959 "
            "0.21727611674823194 -0.13701474018997217,"
            "-3.3639610306789285 0 0.7071067811865475 0 0,"
            "-1.1213203435596428 0 0 0.7071067811865475 0,"
            "0.671336061515156 0.18208981448197611 -0.7875254500568571 "
            "-0.2625084833522857 0.16553835820737872,"
            "5.156617435753728 -1.2321237478911191 -0.7875254500568571 "
            "-0.2625084833522857 0.16553835820737872"),
        wavespeeds_str="4.4 4.4 4.4 6.046329250180534 2.753670749819466",
        char_fluxes_pos_str=("2.2324228059144673 7.380909114098994 "
            "2.4603030380329987 15.885347333048884 0.9465242322295992,"
            "0.5214705489522515 -3.9764501987817145 "
            "-1.3254833995939053 1.3413036144786457 3.767119678640133"),
        char_fluxes_neg_str=("-0.2965647431231918 -0.3514718625761428 "
            "-0.1171572875253809 -1.6097440213843948 -0.5689873221894901,"
            "-0.018976427428922138 1.4911688245431431 "
            "0.49705627484771453 -0.05753157056903602 -1.432380835542713"),
        weno_weights_pos_str=("0.9999999999574652 0.999999999999978 "
            "0.9999999999982254 0.9999999999999919 0.9999999999942413,"
            "3.9384014966385437e-11 2.0284497966079935e-14 "
            "1.6430428067143077e-12 7.542808138536806e-15 5.332240836330361e-12,"
            "3.1507308840273685e-12 1.622759950511315e-15 "
            "1.3144350707803127e-13 6.034246767570432e-16 4.265797494767529e-13"),
        weno_weights_neg_str=("0.9999999386221037 0.9999999999683822 "
            "0.9999999974390363 0.9999999999372101 0.9999999993440752,"
            "5.68308936788955e-8 2.9275831062158654e-11 "
            "2.3712585027633853e-9 5.8138848037057226e-11 6.073372407373299e-10,"
            "4.547002513715645e-9 2.3420726930957915e-12 "
            "1.897052057748541e-10 4.6511252168302e-12 4.8587565862160355e-11"),
        consistent_str="5. 20.4 27. 9. 147.",
        dissipation_pos_str=("3.281326602835503 16.54629350160538 "
            "23.641315459112736 7.8804384863675505 137.3884413587739"),
        dissipation_neg_str=("0.2813265968286516 1.7462934823903076 "
            "2.641315430506613 0.8804384760489334 15.588441251607525"),
        weno_flux_str=("8.562653199664155 38.69258698399568 "
            "53.28263088961935 17.760876962416482 299.9768826103814"),
        direction="y")
single_data["Case c:z"] = FluxDataSingle(
        states_str="2 12 4 8 64,1 3 1 2 11",
        fluxes_str="12 75.2 24 48 403.2,3 10.6 3 6 37.8",
        lam_str=("6. 6. 6. 7.496662954709576 4.503337045290424,"
            "3. 3. 3. 4.496662954709576 1.5033370452904236"),
        R_str=("1 0 0 0.41384589551844686 0.41384589551844686,"
            "4.757359312880716 0 0 2.6759204063286903 1.2617068439555954,"
            "1.5857864376269053 1.4142135623730951 0 0.6562712083807143 "
            "0.6562712083807143,"
            "3.1715728752538106 0 1.4142135623730951 1.3125424167614286 "
            "1.3125424167614286,"
            "17.60303038033002 2.2426406871192857 4.485281374238571 "
            "13.669350884318975 6.941428822961119"),
        R_inv_str=("-1.4118746341171051 0.6518283502446959 "
            "0.21727611674823194 0.4345522334964639 -0.13701474018997217,"
            "-1.1213203435596428 0 0.7071067811865475 0 0,"
            "-2.2426406871192857 0 0 0.7071067811865475 0,"
            "-0.4499842820444868 -0.08041866887030964 -0.2625084833522857 "
            "-0.5250169667045714 0.16553835820737872,"
            "6.27793777931337 -1.4946322312434048 -0.2625084833522857 "
            "-0.5250169667045714 0.16553835820737872"),
        wavespeeds_str="6.6 6.6 6.6 8.246329250180533 4.953670749819467",
        char_fluxes_pos_str=("3.348634208871708 3.690454557049497 "
            "7.380909114098994 26.24407281945101 -0.9962654715332988,"
            "0.7822058234283737 -1.9882250993908581 "
            "-3.9764501987817162 -0.6952990612264269 8.357934000904585"),
        char_fluxes_neg_str=("-0.44484711468478866 -0.1757359312880714 "
            "-0.3514718625761428 -2.447320076091316 -0.8207769392695052,"
            "-0.02846464114338254 0.7455844122715716 "
            "1.4911688245431431 0.8126310150223119 -3.0474996241899346"),
        weno_weights_pos_str=("0.999999999991598 0.9999999999996494 "
            "0.999999999999978 0.9999999999999993 0.9999999999999524,"
            "7.779580658268568e-12 3.2455185423228674e-13 "
            "2.028449796607992e-14 6.408020704124379e-16 4.408056940300416e-14,"
            "6.223673030753551e-13 2.596415558497523e-14 "
            "1.6227599505113154e-15 5.126416626873783e-17 3.526445914956101e-15"),
        weno_weights_neg_str=("0.9999999878747177 0.9999999994941201 "
            "0.9999999999683822 0.9999999999967727 0.9999999999851737,"
            "1.1227070122726888e-8 4.684070887233427e-10 "
            "2.9275831062158654e-11 2.988331869942141e-12 1.372802081812638e-11,"
            "8.982122341016933e-10 3.747296441221644e-11 "
            "2.3420726930957915e-12 2.390667520553204e-13 1.0982436589127136e-12"),
        consistent_str="7.5 42.9 13.5 27. 220.5",
        dissipation_pos_str=("4.921989904281101 36.24738971766925 "
            "11.82065772959958 23.641315459201046 206.08266203865145"),
        dissipation_neg_str=("0.4219899024845117 3.9473897091113366 "
            "1.3206577265155963 2.6413154534736667 23.382662006532225"),
        weno_flux_str=("12.843979806765613 83.09477942678058 "
            "26.641315456115176 53.28263091267471 449.9653240451837"),
        direction="z")

# }}}

# {{{ Input data: Case (d)

single_data["Case d:x"] = FluxDataSingle(
        states_str="1 -1 -2 -3 11,2 -4 -8 -12 64",
        fluxes_str="-1 2.6 2 3 -12.6,-4 11.2 16 24 -134.4",
        lam_str=("-1. -1. -1. 0.49666295470957644 -2.4966629547095764,"
            "-2. -2. -2. -0.5033370452904236 -3.4966629547095764"),
        R_str=("1 0 0 0.41384589551844686 0.41384589551844686,"
            "-1.5857864376269053 0 0 0.05083557280583326 -1.363377989567262,"
            "-3.1715728752538106 1.4142135623730951 0 -1.3125424167614286 "
            "-1.3125424167614286,"
            "-4.757359312880716 0 1.4142135623730951 -1.968813625142143 "
            "-1.968813625142143,"
            "17.60303038033002 -4.485281374238571 -6.727922061357858 "
            "9.184069510080404 11.42671019719969"),
        R_inv_str=("-1.4118746341171051 -0.21727611674823194 "
            "-0.4345522334964639 -0.6518283502446959 -0.13701474018997217,"
            "2.2426406871192857 0 0.7071067811865475 0 0,"
            "3.3639610306789285 0 0 0.7071067811865475 0,"
            "4.035297092194084 0.9696152645388333 0.5250169667045714 "
            "0.7875254500568571 0.16553835820737872,"
            "1.7926564050747988 -0.4445982978342618 0.5250169667045714 "
            "0.7875254500568571 0.16553835820737872"),
        wavespeeds_str="2.2 2.2 2.2 0.553670749819466 3.8463292501805344",
        char_fluxes_pos_str=("0.009488213714461069 0.49705627484771453 "
            "0.7455844122715716 0.31431832174320373 0.4306378813126712,"
            "0.1482823715615963 -0.1171572875253809 "
            "-0.1757359312880714 0.20004041758408686 0.8893252542028551"),
        char_fluxes_neg_str=("-0.26073527447612554 -1.3254833995939053 "
            "-1.9882250993908572 -0.5017887559695788 -2.052422890589809,"
            "-1.1162114029572359 2.4603030380329987 "
            "3.690454557049497 -0.4290108979594782 -7.986924884679773"),
        weno_weights_pos_str=("0.9999990185022956 0.9999999974390363 "
            "0.9999999994941201 0.9999978651320311 0.9999999917661908,"
            "9.08762722250494e-7 2.3712585027633853e-9 "
            "4.684070887233427e-10 1.976628702372801e-6 7.62387333908512e-9,"
            "7.27349821618269e-8 1.897052057748541e-10 "
            "3.747296441221644e-11 1.5823926647977797e-7 6.099359570650648e-10"),
        weno_weights_neg_str=("0.999999999319454 0.9999999999982254 "
            "0.9999999999996494 0.9999870425235151 0.9999999999997061,"
            "6.301345524776077e-10 1.6430428067143077e-12 "
            "3.245518542322869e-13 0.000011996153712066484 2.721049684463335e-13,"
            "5.041138413805998e-11 1.3144350707803127e-13 "
            "2.5964155584975223e-14 9.613227729623656e-7 2.1768403038595738e-14"),
        consistent_str="-2.5 6.9 9. 13.5 -73.5",
        dissipation_pos_str=("-0.14066328824586258 0.42563568844302985 "
            "0.8804384437989112 1.3206576666570364 -7.7942206041633595"),
        dissipation_neg_str=("-1.6406634409601506 4.725635754575325 "
            "7.88043892893644 11.820658393408754 -68.69422377699841"),
        weno_flux_str=("-4.281326729206013 12.051271443018354 "
            "17.76087737273535 26.641316060065794 -149.98844438116177"),
        direction="x")
single_data["Case d:y"] = FluxDataSingle(
        states_str="1 -2 -3 -1 11,2 -8 -12 -4 64",
        fluxes_str="-2 5.6 6 2 -25.2,-8 35.2 48 16 -268.8",
        lam_str=("-2. -2. -2. -0.5033370452904236 -3.4966629547095764,"
            "-4. -4. -4. -2.5033370452904236 -5.496662954709576"),
        R_str=("1 0 0 0.41384589551844686 0.41384589551844686,"
            "-3.1715728752538106 0 0 -0.605435635574881 -2.019649197947976,"
            "-4.757359312880716 1.4142135623730951 0 -1.968813625142143 "
            "-1.968813625142143,"
            "-1.5857864376269053 0 1.4142135623730951 -0.6562712083807143 "
            "-0.6562712083807143,"
            "17.60303038033002 -6.727922061357858 -2.2426406871192857 "
            "8.062749166520762 12.548030540759331"),
        R_inv_str=("-1.4118746341171051 -0.4345522334964639 "
            "-0.6518283502446959 -0.21727611674823194 -0.13701474018997217,"
            "3.3639610306789285 0 0.7071067811865475 0 0,"
            "1.1213203435596428 0 0 0.7071067811865475 0,"
            "5.156617435753728 1.2321237478911191 0.7875254500568571 "
            "0.2625084833522857 0.16553835820737872,"
            "0.671336061515156 -0.18208981448197611 0.7875254500568571 "
            "0.2625084833522857 0.16553835820737872"),
        wavespeeds_str="4.4 4.4 4.4 2.753670749819466 6.046329250180534",
        char_fluxes_pos_str=("0.018976427428922138 1.4911688245431431 "
            "0.49705627484771453 1.432380835542713 0.05753157056903602,"
            "0.2965647431231918 -0.3514718625761428 "
            "-0.1171572875253809 0.5689873221894901 1.6097440213843948"),
        char_fluxes_neg_str=("-0.5214705489522515 -3.9764501987817145 "
            "-1.3254833995939053 -3.767119678640133 -1.3413036144786457,"
            "-2.2324228059144673 7.380909114098994 "
            "2.4603030380329987 -0.9465242322295992 -15.885347333048884"),
        weno_weights_pos_str=("0.9999999386221037 0.9999999999683822 "
            "0.9999999974390363 0.9999999993440752 0.9999999999372101,"
            "5.68308936788955e-8 2.9275831062158654e-11 "
            "2.3712585027633853e-9 6.073372407373299e-10 5.8138848037057226e-11,"
            "4.547002513715645e-9 2.3420726930957915e-12 "
            "1.897052057748541e-10 4.8587565862160355e-11 4.6511252168302e-12"),
        weno_weights_neg_str=("0.9999999999574652 0.999999999999978 "
            "0.9999999999982254 0.9999999999942413 0.9999999999999919,"
            "3.9384014966385437e-11 2.0284497966079935e-14 "
            "1.6430428067143077e-12 5.332240836330361e-12 7.542808138536806e-15,"
            "3.1507308840273685e-12 1.622759950511315e-15 "
            "1.3144350707803127e-13 4.265797494767529e-13 6.034246767570432e-16"),
        consistent_str="-5. 20.4 27. 9. -147.",
        dissipation_pos_str=("-0.2813265968286516 1.7462934823903074 "
            "2.641315430506612 0.8804384760489334 -15.588441251607525"),
        dissipation_neg_str=("-3.281326602835503 16.54629350160538 "
            "23.641315459112736 7.880438486367551 -137.3884413587739"),
        weno_flux_str=("-8.562653199664155 38.69258698399569 "
            "53.28263088961934 17.760876962416486 -299.97688261038144"),
        direction="y")
single_data["Case d:z"] = FluxDataSingle(
        states_str="1 -3 -1 -2 11,2 -12 -4 -8 64",
        fluxes_str="-3 10.6 3 6 -37.8,-12 75.2 24 48 -403.2",
        lam_str=("-3. -3. -3. -1.5033370452904236 -4.496662954709576,"
            "-6. -6. -6. -4.503337045290424 -7.496662954709576"),
        R_str=("1 0 0 0.41384589551844686 0.41384589551844686,"
            "-4.757359312880716 0 0 -1.2617068439555954 -2.6759204063286903,"
            "-1.5857864376269053 1.4142135623730951 0 -0.6562712083807143 "
            "-0.6562712083807143,"
            "-3.1715728752538106 0 1.4142135623730951 -1.3125424167614286 "
            "-1.3125424167614286,"
            "17.60303038033002 -2.2426406871192857 -4.485281374238571 "
            "6.941428822961119 13.669350884318975"),
        R_inv_str=("-1.4118746341171051 -0.6518283502446959 "
            "-0.21727611674823194 -0.4345522334964639 -0.13701474018997217,"
            "1.1213203435596428 0 0.7071067811865475 0 0,"
            "2.2426406871192857 0 0 0.7071067811865475 0,"
            "6.27793777931337 1.4946322312434048 0.2625084833522857 "
            "0.5250169667045714 0.16553835820737872,"
            "-0.4499842820444868 0.08041866887030964 0.2625084833522857 "
            "0.5250169667045714 0.16553835820737872"),
        wavespeeds_str="6.6 6.6 6.6 4.953670749819467 8.246329250180533",
        char_fluxes_pos_str=("0.02846464114338254 0.7455844122715716 "
            "1.4911688245431431 3.0474996241899346 -0.8126310150223119,"
            "0.44484711468478866 -0.1757359312880714 "
            "-0.3514718625761428 0.8207769392695052 2.447320076091316"),
        char_fluxes_neg_str=("-0.7822058234283737 -1.9882250993908581 "
            "-3.9764501987817162 -8.357934000904585 0.6952990612264269,"
            "-3.348634208871708 3.690454557049497 "
            "7.380909114098994 0.9962654715332988 -26.24407281945101"),
        weno_weights_pos_str=("0.9999999878747177 0.9999999994941201 "
            "0.9999999999683822 0.9999999999851737 0.9999999999967727,"
            "1.1227070122726888e-8 4.684070887233427e-10 "
            "2.9275831062158654e-11 1.372802081812638e-11 2.988331869942141e-12,"
            "8.982122341016933e-10 3.747296441221644e-11 "
            "2.3420726930957915e-12 1.0982436589127136e-12 2.390667520553204e-13"),
        weno_weights_neg_str=("0.999999999991598 0.9999999999996494 "
            "0.999999999999978 0.9999999999999524 0.9999999999999993,"
            "7.779580658268568e-12 3.2455185423228674e-13 "
            "2.028449796607992e-14 4.408056940300416e-14 6.408020704124379e-16,"
            "6.223673030753551e-13 2.596415558497523e-14 "
            "1.6227599505113154e-15 3.526445914956101e-15 5.126416626873783e-17"),
        consistent_str="-7.5 42.9 13.5 27. -220.5",
        dissipation_pos_str=("-0.42198990248451174 3.947389709111337 "
            "1.3206577265155963 2.6413154534736667 -23.382662006532225"),
        dissipation_neg_str=("-4.9219899042811 36.247389717669265 "
            "11.820657729599578 23.641315459201046 -206.08266203865145"),
        weno_flux_str=("-12.843979806765612 83.09477942678059 "
            "26.641315456115173 53.28263091267471 -449.9653240451837"),
        direction="z")

# }}}


@pytest.fixture(scope="session", params=[
    "Case a:x", "Case a:y", "Case a:z",
    "Case b:x", "Case b:y", "Case b:z",
    "Case c:x", "Case c:y", "Case c:z",
    "Case d:x", "Case d:y", "Case d:z"])
def flux_test_data_fixture(request):
    return single_data[request.param]


# vim: foldmethod=marker
