__copyright__ = "Copyright (C) 2019 Timothy A. Smith"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import numpy as np
import pytest
import utilities as u
import weno_reference_implementation as ref


class WindowData:
    # FIXME: can we set some of these constants from ref.gas?
    #   -- if all nvars references come from there, it's relatively easy to
    #   introduce a new gas with more (e.g. scalar) variables
    nvars = 5
    ndim = 3
    dirs = {"x": 1, "y": 2, "z": 3}

    def __init__(self, states_str, direction):
        self.direction = self.dirs[direction]
        self.dir_internal = self.direction-1

        # FIXME: Move array_from_string stuff outside WindowResults
        #   -- just pass an array & have external utilities that generate
        #   Riemann, sine wave, etc. initial conditions
        self.state_pair = u.transposed_array_from_string(states_str)
        self.states = u.expand_to_n(self.state_pair, 6)

        # FIXME: these should be generalized fluxes
        # FIXME: make a clear distinction between fluxes in physical and
        # generalized coordinates
        self.fluxes = ref.pointwise_fluxes(
                self.states)[:,:,self.dir_internal].T.copy(order="F")

        self.metrics = np.array([np.identity(self.ndim) for i in range(6)],
                dtype=np.float64, order="F")
        self.jacobians = np.repeat(1.0, 6)


@pytest.fixture(scope="session", params=[
    ("1 1 1 1 5.5,1 1 1 1 5.5", "x"),
    ("1 1 1 1 5.5,1 1 1 1 5.5", "y"),
    ("1 1 1 1 5.5,1 1 1 1 5.5", "z"),
    ("2 4 4 4 20,1 1 1 1 5.5", "x"),
    ("2 4 4 4 20,1 1 1 1 5.5", "y"),
    ("2 4 4 4 20,1 1 1 1 5.5", "z"),
    ("1 -1 -1 -1 5.5,2 -4 -4 -4 20", "x"),
    ("1 -1 -1 -1 5.5,2 -4 -4 -4 20", "y"),
    ("1 -1 -1 -1 5.5,2 -4 -4 -4 20", "z"),
    ("2 4 8 12 64,1 1 2 3 11", "x"),
    ("2 8 12 4 64,1 2 3 1 11", "y"),
    ("2 12 4 8 64,1 3 1 2 11", "z"),
    ("1 -1 -2 -3 11,2 -4 -8 -12 64", "x"),
    ("1 -2 -3 -1 11,2 -8 -12 -4 64", "y"),
    ("1 -3 -1 -2 11,2 -12 -4 -8 64", "z")
    ])
def window_data(request):
    return WindowData(*request.param)


# vim: foldmethod=marker
