__copyright__ = "Copyright (C) 2019 Timothy A. Smith"

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import numpy as np
import pytest
import utilities as u
import weno_reference_implementation as ref


class WindowData:
    nvars = ref.gas.nvars
    ndim = ref.gas.ndim
    dirs = ref.gas.dirs

    def __init__(self, states, direction, window_metrics):
        self.direction = self.dirs[direction]
        self.dir_internal = self.direction-1

        self.states = states
        self.metrics, self.jacobians = window_metrics

        self.fluxes = ref.pointwise_fluxes(self.states)
        self.generalized_fluxes = np.array([np.dot(flux, metrics/jacobian)
                for flux, metrics, jacobian
                in zip(self.fluxes, self.metrics, self.jacobians)]
                )[:,:,self.dir_internal].T.copy(order="F")


@pytest.fixture(scope="session")
def states_generator():
    def states(states_str):
        if states_str == "sine":
            rho = np.sin(0.01*np.arange(6)+1.55)
            vel = np.repeat(np.array([1.0, 2.0, 3.0])[None,:], 6, axis=0)
            pressure = np.repeat(3.2, 6)
            return u.transposed_array([ref.gas.conserved(r, v, p)
                for r, v, p
                in zip(rho, vel, pressure)])
        else:
            state_pair = u.transposed_array_from_string(states_str)
            return u.expand_to_n(state_pair, 6)

    return states


@pytest.fixture(scope="session")
def metrics_generator():
    ndim = ref.gas.ndim

    def window_metrics(metric_str):
        if (metric_str == "uniform"):
            metrics = np.array([np.identity(ndim) for i in range(6)],
                               dtype=np.float64,
                               order="F")
            jacobians = np.repeat(1.0, 6)
        else:
            raise ValueError("Metric string {} not supported".format(metric_str))
        return metrics, jacobians

    return window_metrics


@pytest.fixture(scope="session", params=[
    ("sine", "x", "uniform"),
    ("sine", "y", "uniform"),
    ("sine", "z", "uniform"),
    ("1 1 1 1 5.5,1 1 1 1 5.5", "x", "uniform"),
    ("1 1 1 1 5.5,1 1 1 1 5.5", "y", "uniform"),
    ("1 1 1 1 5.5,1 1 1 1 5.5", "z", "uniform"),
    ("2 4 4 4 20,1 1 1 1 5.5", "x", "uniform"),
    ("2 4 4 4 20,1 1 1 1 5.5", "y", "uniform"),
    ("2 4 4 4 20,1 1 1 1 5.5", "z", "uniform"),
    ("1 -1 -1 -1 5.5,2 -4 -4 -4 20", "x", "uniform"),
    ("1 -1 -1 -1 5.5,2 -4 -4 -4 20", "y", "uniform"),
    ("1 -1 -1 -1 5.5,2 -4 -4 -4 20", "z", "uniform"),
    ("2 4 8 12 64,1 1 2 3 11", "x", "uniform"),
    ("2 8 12 4 64,1 2 3 1 11", "y", "uniform"),
    ("2 12 4 8 64,1 3 1 2 11", "z", "uniform"),
    ("1 -1 -2 -3 11,2 -4 -8 -12 64", "x", "uniform"),
    ("1 -2 -3 -1 11,2 -8 -12 -4 64", "y", "uniform"),
    ("1 -3 -1 -2 11,2 -12 -4 -8 64", "z", "uniform")
    ])
def window_data(request, states_generator, metrics_generator):
    states_str = request.param[0]
    dir_str = request.param[1]
    metric_str = request.param[2]

    return WindowData(states_generator(states_str),
                      dir_str,
                      metrics_generator(metric_str))


class PairData:
    def __init__(self, data):
        self.nvars = data.nvars
        self.ndim = data.ndim
        self.direction = data.direction
        self.dir_internal = data.dir_internal

        self.state_pair = data.states[:,2:4]
        self.generalized_flux_pair = data.generalized_fluxes[:,2:4]

        weighted_metric_slice = data.metrics[2:4]/data.jacobians[2:4,None,None]
        self.frozen_metrics = np.mean(weighted_metric_slice, axis=0)
        self.frozen_jacobian = np.mean(data.jacobians[2:4])
        self.combined_frozen_metrics = np.sum(self.frozen_metrics**2, axis=1)


@pytest.fixture(scope="session")
def pair_data(window_data):
    return PairData(window_data)


# vim: foldmethod=marker
