__copyright__ = """
Copyright (C) 2017 Andreas Kloeckner
"""

__license__ = """
Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
"""

import sys
import numpy as np

from warnings import warn
import pyopencl._cl as _cl
from pytools.persistent_dict import WriteOncePersistentDict
from pytools.py_codegen import Indentation
from pyopencl.tools import _NumpyTypesKeyBuilder

_PYPY = "__pypy__" in sys.builtin_module_names


# {{{ arg packing helpers

_size_t_char = ({
    8: "Q",
    4: "L",
    2: "H",
    1: "B",
})[_cl._sizeof_size_t()]
_type_char_map = {
    "n": _size_t_char.lower(),
    "N": _size_t_char
}
del _size_t_char

# }}}


# {{{ individual arg handling

def generate_buffer_arg_setter(gen, arg_idx, buf_var):
    if _PYPY:
        # https://github.com/numpy/numpy/issues/5381
        gen(f"if isinstance({buf_var}, np.generic):")
        with Indentation(gen):
            if _PYPY:
                gen("{buf_var} = np.asarray({buf_var})".format(buf_var=buf_var))
            else:
                gen("{buf_var} = np.getbuffer({buf_var})".format(buf_var=buf_var))

    gen("""
        self._set_arg_buf({arg_idx}, {buf_var})
        """
        .format(arg_idx=arg_idx, buf_var=buf_var))

# }}}


# {{{ generic arg handling body

def generate_generic_arg_handling_body(num_args):
    from pytools.py_codegen import PythonCodeGenerator
    gen = PythonCodeGenerator()

    if num_args == 0:
        gen("pass")

    for i in range(num_args):
        gen(f"# process argument {i}")
        gen("")
        gen(f"current_arg = {i}")
        gen(f"self.set_arg({i}, arg{i})")
        gen("")

    return gen

# }}}


# {{{ specific arg handling body

def generate_specific_arg_handling_body(function_name,
        num_cl_args, scalar_arg_dtypes,
        work_around_arg_count_bug, warn_about_arg_count_bug,
        include_debug_helpers):

    assert work_around_arg_count_bug is not None
    assert warn_about_arg_count_bug is not None

    fp_arg_count = 0
    cl_arg_idx = 0

    from pytools.py_codegen import PythonCodeGenerator
    gen = PythonCodeGenerator()

    if not scalar_arg_dtypes:
        gen("pass")

    for arg_idx, arg_dtype in enumerate(scalar_arg_dtypes):
        gen(f"# process argument {arg_idx}")
        gen("")
        if include_debug_helpers:
            gen(f"current_arg = {arg_idx}")
        arg_var = "arg%d" % arg_idx

        if arg_dtype is None:
            gen(f"self.set_arg({cl_arg_idx}, {arg_var})")
            cl_arg_idx += 1
            gen("")
            continue

        arg_dtype = np.dtype(arg_dtype)

        if arg_dtype.char == "V":
            gen(f"self.set_arg({cl_arg_idx}, {arg_var})")
            cl_arg_idx += 1

        elif arg_dtype.kind == "c":
            if warn_about_arg_count_bug:
                warn("{knl_name}: arguments include complex numbers, and "
                        "some (but not all) of the target devices mishandle "
                        "struct kernel arguments (hence the workaround is "
                        "disabled".format(
                            knl_name=function_name), stacklevel=2)

            if arg_dtype == np.complex64:
                arg_char = "f"
            elif arg_dtype == np.complex128:
                arg_char = "d"
            else:
                raise TypeError("unexpected complex type: %s" % arg_dtype)

            if (work_around_arg_count_bug == "pocl"
                    and arg_dtype == np.complex128
                    and fp_arg_count + 2 <= 8):
                gen(
                        "buf = pack('{arg_char}', {arg_var}.real)"
                        .format(arg_char=arg_char, arg_var=arg_var))
                gen(f"self._set_arg_buf({cl_arg_idx}, buf)")
                cl_arg_idx += 1
                if include_debug_helpers:
                    gen("current_arg = current_arg + 1000")
                gen(
                        "buf = pack('{arg_char}', {arg_var}.imag)"
                        .format(arg_char=arg_char, arg_var=arg_var))
                gen(f"self._set_arg_buf({cl_arg_idx}, buf)")
                cl_arg_idx += 1

            elif (work_around_arg_count_bug == "apple"
                    and arg_dtype == np.complex128
                    and fp_arg_count + 2 <= 8):
                raise NotImplementedError("No work-around to "
                        "Apple's broken structs-as-kernel arg "
                        "handling has been found. "
                        "Cannot pass complex numbers to kernels.")

            else:
                gen(
                        "buf = pack('{arg_char}{arg_char}', "
                        "{arg_var}.real, {arg_var}.imag)"
                        .format(arg_char=arg_char, arg_var=arg_var))
                gen(f"self._set_arg_buf({cl_arg_idx}, buf)")
                cl_arg_idx += 1

            fp_arg_count += 2

        else:
            if arg_dtype.kind == "f":
                fp_arg_count += 1

            arg_char = arg_dtype.char
            arg_char = _type_char_map.get(arg_char, arg_char)
            gen(
                    "buf = pack('{arg_char}', {arg_var})"
                    .format(
                        arg_char=arg_char,
                        arg_var=arg_var))
            gen(f"self._set_arg_buf({cl_arg_idx}, buf)")
            cl_arg_idx += 1

        gen("")

    if cl_arg_idx != num_cl_args:
        raise TypeError(
            "length of argument list (%d) and "
            "CL-generated number of arguments (%d) do not agree"
            % (cl_arg_idx, num_cl_args))

    return gen

# }}}


# {{{ error handler

def wrap_in_error_handler(body, arg_names):
    from pytools.py_codegen import PythonCodeGenerator, Indentation

    err_gen = PythonCodeGenerator()

    def gen_error_handler():
        err_gen("""
            if current_arg is not None:
                args = [{args}]
                advice = ""
                from pyopencl.array import Array
                if isinstance(args[current_arg], Array):
                    advice = " (perhaps you meant to pass 'array.data' " \
                        "instead of the array itself?)"

                raise _cl.LogicError(
                        "when processing argument #%d (1-based): %s%s"
                        % (current_arg+1, str(e), advice))
            else:
                raise
            """
            .format(args=", ".join(arg_names)))
        err_gen("")

    err_gen("try:")
    with Indentation(err_gen):
        err_gen.extend(body)
    err_gen("except TypeError as e:")
    with Indentation(err_gen):
        gen_error_handler()
    err_gen("except _cl.LogicError as e:")
    with Indentation(err_gen):
        gen_error_handler()

    return err_gen

# }}}


def _generate_enqueue_and_set_args_module(function_name,
        num_passed_args, num_cl_args,
        scalar_arg_dtypes,
        work_around_arg_count_bug, warn_about_arg_count_bug,
        include_debug_helpers):

    from pytools.py_codegen import PythonCodeGenerator, Indentation

    arg_names = ["arg%d" % i for i in range(num_passed_args)]

    if scalar_arg_dtypes is None:
        body = generate_generic_arg_handling_body(num_passed_args)
    else:
        body = generate_specific_arg_handling_body(
                function_name, num_cl_args, scalar_arg_dtypes,
                warn_about_arg_count_bug=warn_about_arg_count_bug,
                work_around_arg_count_bug=work_around_arg_count_bug,
                include_debug_helpers=include_debug_helpers)

    if include_debug_helpers:
        body = wrap_in_error_handler(body, arg_names)

    gen = PythonCodeGenerator()

    gen("from struct import pack")
    gen("from pyopencl import status_code")
    gen("import numpy as np")
    gen("import pyopencl._cl as _cl")
    gen("")

    # {{{ generate _enqueue

    enqueue_name = "enqueue_knl_%s" % function_name
    gen("def %s(%s):"
            % (enqueue_name,
                ", ".join(
                    ["self", "queue", "global_size", "local_size"]
                    + arg_names
                    + ["global_offset=None",
                        "g_times_l=None",
                        "allow_empty_ndrange=False",
                        "wait_for=None"])))

    with Indentation(gen):
        gen.extend(body)

        gen("""
            return _cl.enqueue_nd_range_kernel(queue, self, global_size, local_size,
                    global_offset, wait_for, g_times_l=g_times_l,
                    allow_empty_ndrange=allow_empty_ndrange)
            """)

    # }}}

    # {{{ generate set_args

    gen("")
    gen("def set_args(%s):"
            % (", ".join(["self"] + arg_names)))

    with Indentation(gen):
        gen.extend(body)

    # }}}

    return gen.get_picklable_module(), enqueue_name


invoker_cache = WriteOncePersistentDict(
        "pyopencl-invoker-cache-v20",
        key_builder=_NumpyTypesKeyBuilder())


def generate_enqueue_and_set_args(function_name,
        num_passed_args, num_cl_args,
        scalar_arg_dtypes,
        work_around_arg_count_bug, warn_about_arg_count_bug):

    cache_key = (function_name, num_passed_args, num_cl_args,
            scalar_arg_dtypes,
            work_around_arg_count_bug, warn_about_arg_count_bug,
            not sys.flags.optimize)

    from_cache = False

    try:
        result = invoker_cache[cache_key]
        from_cache = True
    except KeyError:
        pass

    if not from_cache:
        result = _generate_enqueue_and_set_args_module(*cache_key)
        invoker_cache.store_if_not_present(cache_key, result)

    pmod, enqueue_name = result

    return (
            pmod.mod_globals[enqueue_name],
            pmod.mod_globals["set_args"])

# }}}


# vim: foldmethod=marker
